#include <ifractal.h>


#define BASE64_LOOKUP		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"


// ////////////////////////////////////////////////////////////////////////
_PUBLIC int base64_get_encodedLen(_IN int length)
{
	unsigned int uiRet = (unsigned int) ((length * 8) / 6);

	while ((uiRet % 4) != 0)
		uiRet++;

	return(uiRet); 
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC int b64_encode(_IN const unsigned char *inbuf, _IN size_t length, _OUT char **outbuf)
{
	unsigned int i, outlen, uiOutProg = 0;
	unsigned char bNewLetter = 0, bInProg = 0;
	char *tmpbuf;

	if ((inbuf == NULL) || (outbuf == NULL) || (length < 1))
		return(-1);

	outlen = base64_get_encodedLen(length);
	tmpbuf = if_malloc(outlen + 1);

	for (i = 0 ; i < length ; i++)
	{
		bNewLetter |= (inbuf[i] & ((0xFC << bInProg) & 0xFF)) >> bInProg;
		bNewLetter >>= 2;
		tmpbuf[uiOutProg++] = BASE64_LOOKUP[bNewLetter];

		bNewLetter = 0; // starting over
		bInProg = 8 - (2 + bInProg);
		bNewLetter = (inbuf[i] & (0xFF >> bInProg)) << bInProg;
		bInProg = 8 - bInProg;

		if (bInProg == 6)
		{
			bNewLetter >>= 2;
			tmpbuf[uiOutProg++] = BASE64_LOOKUP[bNewLetter];
			bInProg = 0;
			bNewLetter = 0;
		}
	}

	if (bInProg > 0)
	{
		bNewLetter >>= 2;
		tmpbuf[uiOutProg++] = BASE64_LOOKUP[bNewLetter];
	}

	while ((uiOutProg % 4) != 0)
		tmpbuf[uiOutProg++] = '=';

	tmpbuf[uiOutProg] = '\0';

	*outbuf = tmpbuf;

	return(outlen);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
void b64_decodeblock(unsigned char *in, unsigned char *out)
{	 
	out[0] = (unsigned char ) (in[0] << 2 | in[1] >> 4);
	out[1] = (unsigned char ) (in[1] << 4 | in[2] >> 2);
	out[2] = (unsigned char ) (((in[2] << 6) & 0xc0) | in[3]);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
int b64_decode(_IN char *src, _OUT unsigned char **outdata)
{
	unsigned char dtable[256], *out, *pos, block[4], tmp;
	size_t i, count, olen;
	size_t len = strlen(src);
	int pad = 0;

	memset(dtable, 0x80, 256);
	for (i = 0 ; i < sizeof(BASE64_LOOKUP) - 1 ; i++)
		dtable[(unsigned char) BASE64_LOOKUP[i]] = (unsigned char) i;
	dtable['='] = 0;

	count = 0;
	for (i = 0 ; i < len ; i++)
	{
		if (dtable[(unsigned char) src[i]] != 0x80)
			count++;
	}

	if (count == 0 || count % 4)
		return(-1);

	olen = 3 * count / 4;
	pos = out = if_malloc(olen);
	if (out == NULL)
		return(-2);

	count = 0;
	for (i = 0 ; i < len ; i++)
	{
		tmp = dtable[(unsigned char) src[i]];
		if (tmp == 0x80)
			continue;

		if (src[i] == '=')
			pad++;
		block[count] = tmp;
		count++;
		if (count == 4)
		{
			*pos++ = (block[0] << 2) | (block[1] >> 4);
			*pos++ = (block[1] << 4) | (block[2] >> 2);
			*pos++ = (block[2] << 6) | block[3];
			count = 0;
			if (pad)
			{
				if (pad == 1)
					pos--;
				else if (pad == 2)
					pos -= 2;
				else
				{
					// Invalid padding
					if_free(out);
					return(-3);
				}

				break;
			}
		}
	}

	*outdata = out;
	return(pos - out);
}
// ////////////////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////////////////// //
unsigned char * hex2bin(char *hex_str)
{
	unsigned char *bin, byte;
	int i, len;

	if (hex_str == NULL)
		return(NULL);

	len = strlen(hex_str) / 2;

	bin = MALLOC(len);
	for (i = 0 ; i < len ; i++)
	{
		hex_str[2 * i] = toupper(hex_str[2 * i]);
		hex_str[2 * i + 1] = toupper(hex_str[2 * i + 1]);

		if (hex_str[2 * i] < 'A')
			byte = 16 * (hex_str[2 * i] - '0');
		else
			byte = 16 * (hex_str[2 * i] - 'A' + 10);
		
		if (hex_str[2 * i + 1] < 'A')
			byte += hex_str[2 * i + 1] - '0';
		else
			byte += hex_str[2 * i + 1] - 'A' + 10;

		bin[i] = byte;
	}

	return(bin);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * bin2hex(_IN unsigned char *data, size_t len)
{
	char *buf, *p;
	int i = 0;

	buf = if_malloc(2 * len + 1);
	p = buf;

	for (i = 0 ; i < len ; i++)
		p += snprintf(p, 5, "%02X", data[i]);

	return(buf);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int if_rawdup(char *txt, unsigned char **bin)
{
	int l = strlen(txt);
	*bin = (unsigned char *) if_strdup(txt);
	return(l);
}	
// ///////////////////////////////////////////////////////////////////// //
int if_hex2bin(char *txt, unsigned char **bin)
{
	int l = strlen(txt) / 2;
	*bin = hex2bin(txt);
	return(l);
}
// ///////////////////////////////////////////////////////////////////// //
int if_str_decoder(char *txt, unsigned char **bin, char *format)
{
	struct
	{
		char *format;
		int (*func)(char *txt, unsigned char **bin);
	} decoder[] = {
		{TXT_ENCODE_RAW, if_rawdup},
		{TXT_ENCODE_B64, b64_decode},
		{TXT_ENCODE_HEX, if_hex2bin},
		{NULL, NULL}
	};

	*bin = NULL;

	for (int i = 0 ; decoder[i].format != NULL ; i++)
		if (strcmp(decoder[i].format, format) == 0)
			return(decoder[i].func(txt, bin));

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int if_rawdup2(const unsigned char *bin, size_t len, char **txt)
{
	char *out = if_malloc(len + 1);
	memcpy(out, bin, len);
	*(out + len) = 0;
	*txt = out;
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int if_bin2hex(const unsigned char *bin, size_t len, char **txt)
{
	*txt = bin2hex((unsigned char *) bin, len);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int if_str_encoder(unsigned char *bin, size_t len, char **txt, char *format)
{
	struct
	{
		char *format;
		int (*func)(const unsigned char *bin, size_t, char **txt);
	} decoder[] = {
		{TXT_ENCODE_RAW, if_rawdup2},
		{TXT_ENCODE_B64, b64_encode},
		{TXT_ENCODE_HEX, if_bin2hex},
		{NULL, NULL}
	};

	*txt = NULL;

	for (int i = 0 ; decoder[i].format != NULL ; i++)
		if (strcmp(decoder[i].format, format) == 0)
			return(decoder[i].func(bin, len, txt));

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //



#ifdef BASE64_DEBUG
// ////////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	unsigned char *bufin;
	int outlen, len;
	char *bufout;

	if (argc < 2)
	{
		fprintf(stderr, "\nEncode/decode base64\n\n");
		fprintf(stderr, "\nUso:\n\tshell$ %s e|d\n\n", argv[0]);
		return(1);
	}

	len = cgi_get_content(stdin, (char **) &bufin);

	if (argv[1][0] != 'e')
	{
		outlen = b64_decode((char *) bufin, (unsigned char **) &bufout);
		fwrite(bufout, 1, outlen, stdout);
	}
	else
	{
		outlen = base64_get_encodedLen(len);
		b64_encode(bufin, len, &bufout);
		fprintf(stdout, "%s", bufout);
	}

	fprintf(stderr, "(%d,%d)\n", len, outlen);

	if_free(bufout);
	if_free(bufin);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
#endif

