#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include <ifractal.h>


// //////////////////////////////////////////////////////////////////////
_CALLBACK int delete_fs(DIR_CONTEXT *ctx, FILE_CONTEXT *fctx, void *user_data)
{
	DELETE_CTX *del = (DELETE_CTX *) user_data;
	int dias, r;

	dias = (time(NULL) - fctx->ctime) / (24 * 3600);

	if ((strcmp(fctx->name, ".") == 0) || (strcmp(fctx->name, "..") == 0))
		return(0);

	if ((fctx->type == FILE_TYPE_DIR) && (del->recursivo))
		execute_delete_in(fctx->filename, del);

	if (dias < del->qtdias)
		return(0);

	if (!fs_file_check(fctx, del->prefixo, del->sufixo))
		return(0);

	if (del->test)
	{
		verbose(stderr, "'%s' a ser removido (test).\n", fctx->filename);
		r = 0;
	}
	else
	{
#ifdef WIN32
		if (DeleteFile(fctx->filename)) r = 0; else r = 1;
#else
		r = remove(fctx->filename);
#endif
	}

	if (!r)
	{
		del->apagados++;

		if (del->callback != NULL)
			del->callback(fctx->path, fctx->name, del->user_data);
	}
	else
		verbose(stderr, "Falha ao tentar remover '%s'\n", fctx->filename);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int execute_delete_in(char *dirnome, DELETE_CTX *ctx)
{
	DIR_CONTEXT *dir;

	if((ctx->prefixo[0] == 0) && (ctx->sufixo[0] == 0))
	{
		fprintf(stderr, "|DELETE| E necessario definir o prefixo e/ou sufixo.\n");
		return(-1);
	}

	dir = fs_dir_new(dirnome);
	if (dir == NULL)
	{
		fprintf(stderr, "|DELETE| Diretorio nao encontrado: '%s'.\n", dirnome);
		return(-2);
	}

	fs_dir_iter(dir, delete_fs, ctx);

	fs_dir_free(dir);

	return(ctx->apagados);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
DELETE_CTX * delete_new(char *dirnome, char *prefixo, char *sufixo, int qtdias, int recursivo, DELETE_CALLBACK cb, void *user_data, int test)
{
	DELETE_CTX *del = if_malloc(sizeof(DELETE_CTX));

	del->prefixo = prefixo;
	del->sufixo = sufixo;
	del->qtdias = qtdias;
	del->recursivo = recursivo;
	del->callback = cb;
	del->user_data = user_data;
	del->apagados = 0;
	del->test = test;

	return(del);
}
// //////////////////////////////////////////////////////////////////////
void delete_free(DELETE_CTX *del)
{
	if_free(del);
}
// //////////////////////////////////////////////////////////////////////
int delete_perform(DELETE_CTX *del, char *dir)
{
	int r = execute_delete_in(dir, del);
	return(r);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC int execute_delete(char *dirnome, char *prefixo, char *sufixo, int qtdias, int recursivo, DELETE_CALLBACK cb, void *user_data)
{
	DELETE_CTX *del = delete_new(dirnome, prefixo, sufixo, qtdias, recursivo, cb, user_data, 0);
	int r = execute_delete_in(dirnome, del);
	delete_free(del);
	return(r);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC int execute_delete_test(char *dirnome, char *prefixo, char *sufixo, int qtdias, int recursivo, DELETE_CALLBACK cb, void *user_data)
{
	DELETE_CTX *del = delete_new(dirnome, prefixo, sufixo, qtdias, recursivo, cb, user_data, 1);
	int r = execute_delete_in(dirnome, del);
	delete_free(del);
	return(r);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PUBLIC int delete_get_new_old(_IN char *dirnome, _IN char *prefixo, _IN char *sufixo, _OUT time_t *newer, _OUT time_t *older)
{
	int arquivo_len, sufixo_len;
	char path[PATH_LEN];
	struct dirent *item;
#ifdef WIN32
	struct _stat file;
#else
	struct stat file;
	int status;
#endif
	DIR *dir;

	*newer = 0;
	*older = time(NULL);

	if ((dir = opendir(dirnome)) == NULL)
	{
		fprintf(stderr, "|DELETE| Diretorio nao encontrado: '%s'.\n", dirnome);
		return(1);
	}

	while ((item = readdir(dir)) != NULL)
	{
		snprintf(path, PATH_LEN, "%s%c%s", dirnome, PATH_SEPARATOR, item->d_name);

#ifdef WIN32
		_stat(path, &file);
		if (S_ISDIR(file.st_mode))
			continue;
#else
		status = lstat(path, &file);
		if(status == -1)
			continue;

		if(item->d_type != DT_REG)
			continue;
#endif

		if((strcmp(item->d_name, ".") == 0) || (strcmp(item->d_name, "..") == 0))
			continue;

		if (strncmp(item->d_name, prefixo, strlen(prefixo)) != 0)
			continue;

		sufixo_len = strlen(sufixo);
		arquivo_len = strlen(item->d_name);

		if(strcmp(item->d_name + arquivo_len - sufixo_len, sufixo) != 0)
			continue;

		if (file.st_ctime > *newer)
			*newer = file.st_ctime;

		if (file.st_ctime < *older)
			*older = file.st_ctime;
	}
	closedir(dir);

	return(0);
}
// //////////////////////////////////////////////////////////////////////



#ifdef STANDALONE

int delete_callback(char *path, char *filename, void *user_data)
{
	printf("%s%c%s - removido\n", path, PATH_SEPARATOR, filename);
	return(0);
}

#define DESCRIPT	"Deleta os arquivos recursivamente por idade.\n"

IF_GETOPT configs[] = {
	{0, 't', IF_GETOPT_TYPE_NONE, "test", ".", 0, "Lista arquivos sem deletar."},
	{0, 'd', IF_GETOPT_TYPE_STRING, "dir", ".", 0, "Diretorio de referencia."},
	{0, 'r', IF_GETOPT_TYPE_NONE, "recursivo", "", 0, "Deletar recursivamente."},
	{0, 'i', IF_GETOPT_TYPE_NUMBER, "idade", "1", 0, "Idade dos arquivos a serem deletados em dias (default 1)."},
	{0, 'p', IF_GETOPT_TYPE_STRING, "prefixo", "", 0, "Prefixo do nome dos arquivos a serem deletados. (REGEX quando sufixo for vazio)"},
	{0, 's', IF_GETOPT_TYPE_STRING, "sufixo", "", 0, "Sufixo do nome dos arquivos a serem deletados."},
	{0, 0, 0, 0, 0, 0, 0}
};

int main(int argc,char *argv[])
{
	char *dirnome;
	int qtdias;
	char *prefixo;
	char *sufixo;
	time_t newer, older;
	struct tm *dt;
	int r;

	r = if_getopt(configs, argc, argv);
	if (	(r < 0) || 
		(!if_getopt_isChecked(configs, "prefixo") && !if_getopt_isChecked(configs, "sufixo"))
		)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -i <idade> em dias -s <sufixo>\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\tshell$ %s -i 2 -s .log\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

	dirnome = if_getopt_getValue(configs, "dir");
	qtdias = atoi(if_getopt_getValue(configs, "idade"));
	prefixo = if_getopt_getValue(configs, "prefixo");
	sufixo = if_getopt_getValue(configs, "sufixo");

	delete_get_new_old(dirnome, prefixo, sufixo, &newer, &older);
	dt = localtime(&newer);
	fprintf(stdout, "Mais recente: %02d/%02d/%d %02d:%02d:%02d\n",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min, dt->tm_sec);

	dt = localtime(&older);
	fprintf(stdout, "Mais antigo: %02d/%02d/%d %02d:%02d:%02d\n",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min, dt->tm_sec);

	int rec = if_getopt_isChecked(configs, "recursivo");

	if (if_getopt_isChecked(configs, "test"))
		r = execute_delete_test(dirnome, prefixo, sufixo, qtdias, rec, delete_callback, NULL);
	else
		r = execute_delete(dirnome, prefixo, sufixo, qtdias, rec, delete_callback, NULL);

	printf("Foram deletados %d arquivos.\n", r);
	return(0);
}

#endif
