#include <ifractal.h>
#include <netraw.h>
#include <diagnostico.h>

#ifdef WIN32
#include <shlobj.h>
#endif

#include "diagnostico_html.h"



// //////////////////////////////////////////////////////////////////////
int diagnostico_server_identification(JSON_VALUE *jhost, char *server)
{
	struct
	{
		char *name;
		int id;
	} headers[] = { 
	{"GOAHEAD-WEBS", DIAGNOSTICO_ID_IPCAM_DLINK},
	{"ANDROID WEBCAM", DIAGNOSTICO_ID_IPCAM_ANDROID},
	{"NETWAVE IP CAMERA", DIAGNOSTICO_ID_IPCAM_FOSCAM},
	{"REP SERVER", DIAGNOSTICO_ID_HENRY8x},
	{"ARGOS SERVER", DIAGNOSTICO_ID_HENRY8x}, 
	{"ROUTER WEBSERVER", DIAGNOSTICO_ID_ROUTER_TP_LINK},
	{"INTOTO HTTP SERVER", DIAGNOSTICO_ID_ROUTER_CISCO}, 
	{"VERTICAL_IP_SECURE_PLATFORM_VSP", DIAGNOSTICO_ID_ROUTER_CISCO},
	{"SIYOU SERVER", DIAGNOSTICO_ID_ROUTER_DLINK},
	{"HTTPD", DIAGNOSTICO_ID_ROUTER_DLINK},
	{"VIRATA-EMWEB", DIAGNOSTICO_ID_HP_PRINTER},
	{"UNIX", DIAGNOSTICO_ID_UNIX},
	{"LINUX", DIAGNOSTICO_ID_UNIX},
	{"UBUNTU", DIAGNOSTICO_ID_UBUNTU},
	{"RED HAT", DIAGNOSTICO_ID_REDHAT},
	{"GOOGLE SEARCH APPLIANCE", DIAGNOSTICO_ID_GSA},
	{"MICROSOFT", DIAGNOSTICO_ID_WINDOWS},
	{"IIS", DIAGNOSTICO_ID_WINDOWS},
	{"WMI", DIAGNOSTICO_ID_WINDOWS},
	{"WIN", DIAGNOSTICO_ID_WINDOWS},
	{"HING A SECURE INTEGRATED LIGHTS OUT SESSION", DIAGNOSTICO_ID_ORACLE},
	{"HEXA", DIAGNOSTICO_ID_HEXA},
	{NULL, -1}};
	int i;

	for (i = 0 ; server[i] != 0 ; i++)
		server[i] = toupper(server[i]);

	for (i = 0 ; headers[i].name != NULL ; i++)
	{
		if (strstr(server, headers[i].name) != NULL)
			return(headers[i].id);
	}

	return(-1);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
char * diagnostico_get_ID_descript(DIAGNOSTICO_ID id)
{
	switch (id)
	{
		case DIAGNOSTICO_ID_UNKNOWN: return("Nao identificado");
		case DIAGNOSTICO_ID_IFRACTAL: return("Modulo iFractal SIIN");
		case DIAGNOSTICO_ID_IFRACTAL_OLD: return("Modulo iFractal (antigo)");
		case DIAGNOSTICO_ID_IPCAM_FOSCAM: return("IPCAM Foscam");
		case DIAGNOSTICO_ID_IPCAM_ANDROID: return("IPCAM Android");
		case DIAGNOSTICO_ID_IPCAM_DLINK: return("IPCAM Dlink");
		case DIAGNOSTICO_ID_ROUTER_TP_LINK: return("Router TP-link");
		case DIAGNOSTICO_ID_ROUTER_DLINK: return("Router Dlink");
		case DIAGNOSTICO_ID_ROUTER_CISCO: return("Router Cisco");
		case DIAGNOSTICO_ID_TIBBO: return("Tibbo");
		case DIAGNOSTICO_ID_HENRY7x: return("Henry7x");
		case DIAGNOSTICO_ID_HENRY8x: return("Henry8x");
		case DIAGNOSTICO_ID_HP_PRINTER: return("Impressora HP");
		case DIAGNOSTICO_ID_UBUNTU: return("GNU/Linux Ubuntu");
		case DIAGNOSTICO_ID_REDHAT: return("GNU/Linux Red Hat");
		case DIAGNOSTICO_ID_GSA: return("Google Search Appliance");
		case DIAGNOSTICO_ID_UNIX: return("Unix");
		case DIAGNOSTICO_ID_WINDOWS: return("Windows");
		case DIAGNOSTICO_ID_ORACLE: return("Oracle");
		case DIAGNOSTICO_ID_HEXA: return("Henry - Hexa");
		case DIAGNOSTICO_ID_LEN: break;
	}

	return("---");
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
char * diagnostico_get_STATUS_descript(DIAGNOSTICO_STATUS status)
{
	switch (status)
	{
		case DIAGNOSTICO_STATUS_IRRELEVANT: return("Irrelevante");
		case DIAGNOSTICO_STATUS_OK: return("OK");
		case DIAGNOSTICO_STATUS_CONFLICT: return("Conflito");
		case DIAGNOSTICO_STATUS_UNREACHABLE: return("Sem comunicacao");
		case DIAGNOSTICO_STATUS_MODULE_UNREACHABLE: return("Sem comunicacao com modulo");
		case DIAGNOSTICO_STATUS_MODULE_FAIL: return("Erro na configuracao");
		case DIAGNOSTICO_STATUS_SIIN_UNREACHABLE: return("Sem comunicacao com SIIN");
		case DIAGNOSTICO_STATUS_UNREF: return("Nao cadastrado");
		case DIAGNOSTICO_STATUS_LEN: break;
	}

	return("---");
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
char * diagnostico_get_default_service(int port)
{
	struct
	{
		int port;
		char *service;
	} servs[] = {
		{22, "SSH"},
		{80, "HTTP"},
		{139, "Samba/Netbios"},
		{443, "HTTPS"},
		{2051, "Vetronic"},
		{3000, "Henry"},
		{5432, "PostGreSQL"},
		{5900, "VNC"},
		{5901, "VNC"},
		{6000, "Xserver"},
		{8080, "HTTP/Proxy"},
		{9000, "iFractal SIIN"},
		{10000, "Linux Webmin"},
		{12000, "SIIN"},
		{51000, "Topdata"},
		{0, NULL}};
	int i;

	for (i = 0 ; servs[i].port != 0 ; i++)
		if (servs[i].port == port)
			return(servs[i].service);

	return("---");
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_find_port_iter(JSON_VALUE *json, void *user_data)
{
	void **params = (void **) user_data;
	int *port = (int *) params[0];
	JSON_VALUE **jport = (JSON_VALUE **) params[1];
	JSON_VALUE *aux = json_object_find(json, "port");

	if (aux == NULL)
		return(0);

	if (json_get_int(aux) == *port)
	{
		*jport = json;
		return(1);
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_find_port(_INOUT JSON_VALUE *jhost, _IN int port)
{
	JSON_VALUE *jport = NULL, *jports;
	void *params[] = {&port, &jport};

	if (jhost == NULL)
		return(NULL);

	jports = json_object_find(jhost, "ports");
	if (jports == NULL)
		return(NULL);

	json_array_iter(jports, diagnostico_find_port_iter, params);

	return(jport);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_get_port_from_id_iter(JSON_VALUE *json, void *user_data)
{
	void **params = (void **) user_data;
	int *id = (int *) params[0];
	char **port = (char **) params[1];
	JSON_VALUE *aux = json_object_find(json, "id");

	if (aux == NULL)
		return(0);

	if (json_get_int(aux) == *id)
	{
		aux = json_object_find(json, "port");
		if (aux != NULL)
			*port = json_get_string(aux);

		return(1);
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
char * diagnostico_get_port_from_id(_INOUT JSON_VALUE *jhost, _IN DIAGNOSTICO_SERVICE_ID id)
{
	JSON_VALUE *jports;
	char *port = NULL;
	void *params[] = {&id, &port};

	if (jhost == NULL)
		return(NULL);

	jports = json_object_find(jhost, "ports");
	if (jports == NULL)
		return(NULL);

	json_array_iter(jports, diagnostico_get_port_from_id_iter, params);

	return(port);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_add_port(
	_INOUT JSON_VALUE *jhost,
	_IN int port,
	_IN char *server)
{
	char *list[] = {
		"id", "0",
		"default", diagnostico_get_default_service(port),
		"server", server,
		NULL, NULL};
	JSON_VALUE *aux, *jport = NULL, *jports;

	jports = json_object_find(jhost, "ports");
	if (jports == NULL)
	{
		jports = json_array_new(1);
		json_object_add(jhost, "ports", jports);
	}

	jport = diagnostico_find_port(jhost, port);
	if (jport == NULL)
	{
		jport = json_object_new_list(list);
		json_object_add(jport, "port", json_integer_new(port));
		json_array_add(jports, jport);

		return(jport);
	}

	aux = json_object_find(jport, "server");
	if ((aux == NULL) || (json_get_string(aux)[0] == 0))
		json_object_add(jport, "server", json_string_new(server));

	return(jport);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_find_host_iter(JSON_VALUE *json, void *user_data)
{
	void **params = (void **) user_data;
	char *ip = (char *) params[0];
	JSON_VALUE **jhost = (JSON_VALUE **) params[1];
	JSON_VALUE *aux;

	aux = json_object_find(json, "ip");
	if (aux == NULL)
		return(0);

	if (strcmp(ip, json_get_string(aux)) == 0)
	{
		*jhost = json;
		return(1);
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_find_host(_INOUT JSON_VALUE *jhosts, _IN char *ip)
{
	JSON_VALUE *jhost = NULL;
	void *params[] = {ip, &jhost};

	json_array_iter(jhosts, diagnostico_find_host_iter, params);

	return(jhost);
}
// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_add_host(
	_INOUT JSON_VALUE *jhosts, 
	_IN char *ip, 
	_IN char *name, 
	_IN char *orig)
{
	char *list[] = {
		"ip", ip, 
		"name", name,
		"origem", orig,
		"mac", "XXXX",
		"id","0",
		"status","0",
		NULL, NULL};
	JSON_VALUE *jhost;

	jhost = diagnostico_find_host(jhosts, ip);
	if (jhost == NULL)
	{
		jhost = json_object_new_list(list);
		json_array_add(jhosts, jhost);
	}

	return(jhost);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_get_ssh_server(_IN char *ip, _IN int port, _INOUT char *server, size_t namelen)
{
	char buf[PATH_LEN];
	int sock, n, i;

	snprintf(buf, PATH_LEN, "%d", port);
	sock = openTCP(ip, buf, 3);
	if (sock < 1)
		return(-1);

	n = read_timeout(sock, buf, PATH_LEN, 2);
	if (n > 3)
	{
		for (i = 0 ; i < n ; i++)
			if ((buf[i] < ' ') || (buf[i] > '}'))
			{
				buf[i] = 0;
				break;
			}

		strncpy(server, buf, namelen);
	}

	if_closesocket(sock);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_get_http_server(_IN char *ip, _IN int port, _INOUT char *server, size_t namelen)
{
	char header[] = "Server: ";
	int len = strlen(header);
	char buf[PATH_LEN];
	int sock, n;

	snprintf(buf, PATH_LEN, "%d", port);
	sock = openTCP(ip, buf, 3);
	if (sock < 1)
		return(-1);

	n = snprintf(buf, PATH_LEN, HTTP_GET_HEADERS, "", ip);
	if (send_bytes(sock, buf, n, 1) != n)
		return(-2);

	while (1)
	{
		n = readln_timeout(sock, buf, PATH_LEN, 2);
		if (n < 2)
			break;

		buf[n - 2] = 0;
		if (strncmp(buf, header, len) == 0)
		{
			n = strlen(buf);
			strncpy(server, buf + len, namelen);
			break;
		}
	}

	if_closesocket(sock);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_parse_tibbo(_INOUT JSON_VALUE *jhosts, _IN char *ip, char *data, size_t datalen)
{
	char bufmac[15] = "XXXXXXXXXXXX";
	JSON_VALUE *jhost, *jport;
	char *mac[7];
	char *tk[4];
	int i, port;

	i = tokenizer('/', data, tk, 4);
	if (i < 3)
		return(-1);

	port = atoi(tk[1]);

	jhost = diagnostico_add_host(jhosts, ip, tk[2], "tibbo");
	i = tokenizer('.', tk[0], mac, 7);
	if (i >= 6)
	{
		snprintf(bufmac, sizeof(bufmac), "%02X%02X%02X%02X%02X%02X",
			atoi(mac[0]), atoi(mac[1]), atoi(mac[2]),
			atoi(mac[3]), atoi(mac[4]), atoi(mac[5]));
	}

	json_object_add(jhost, "mac", json_string_new(bufmac));

	if (port == 3000)
	{
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_HENRY7x));
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_UNREF));

		jport = diagnostico_add_port(jhost, port, "Henry 7x");
		json_object_add(jport, "id", json_integer_new(DIAGNOSTICO_SERVICE_ID_HENRY7x));
	}
	else
	{
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_TIBBO));
		diagnostico_add_port(jhost, port, "");
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_parse_siin_foscam(_INOUT JSON_VALUE *jhosts, _IN char *ip, BROADCAST_DATA *data, size_t datalen)
{
	int port = data->foscam.port[0] * 256 + data->foscam.port[1];
	JSON_VALUE *jhost, *jport;

	jhost = diagnostico_add_host(jhosts, ip, data->foscam.name, "siin");
	json_object_add(jhost, "mac", json_string_new(data->foscam.mac));
	json_object_add(jhost, "name", json_string_new(data->foscam.name));

	if (strstr(data->foscam.name, "SIIN") != NULL)
	{
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_IFRACTAL));
		jport = diagnostico_add_port(jhost, port, "SIIN Server");
		json_object_add(jport, "id", json_integer_new(DIAGNOSTICO_SERVICE_ID_SIIN));
	}
	else
	{
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_IPCAM_FOSCAM));
		//json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_UNREF));

		jport = diagnostico_add_port(jhost, port, "");
		json_object_add(jport, "id", json_integer_new(DIAGNOSTICO_SERVICE_ID_HTTP));
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_parse_tcp(_INOUT JSON_VALUE *jhosts, _IN char *ip, void *data, size_t datalen)
{
	JSON_VALUE *jhost;
	struct packet
	{
		struct ip ip;
		struct tcphdr tcphdr;
	} *pkt = (struct packet *) data;
	int sport, dport;

	if (pkt->tcphdr.th_flags != 18) 
		return(-1);

	dport = ((pkt->tcphdr.th_dport & 0xFF) << 8) + (pkt->tcphdr.th_dport >> 8);
	if (dport != 1025)
		return(-2);

	sport = ((pkt->tcphdr.th_sport & 0xFF) << 8) + (pkt->tcphdr.th_sport >> 8);

	jhost = diagnostico_add_host(jhosts, ip, ip, "portscan");
	diagnostico_add_port(jhost, (int) sport, "");

	if (sport == 9000)
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_IFRACTAL_OLD));

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_parse_icmp(_INOUT JSON_VALUE *jhosts, _IN char *ip, void *data, size_t datalen)
{
	JSON_VALUE *jhost;
	struct packet
	{
		struct ip ip;
		struct icmphdr icmp;
	} *pkt = (struct packet *) data;

	if (pkt->icmp.type == 3)
		return(-1);

	jhost = diagnostico_add_host(jhosts, ip, ip, "ping");

#ifdef __linux__
	struct addrinfo hints, *res;
	char addrstr[100];
	void *ptr;
	int err;

	memset (&hints, 0, sizeof (hints));
	hints.ai_family = AF_INET;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags |= AI_CANONNAME;

	err = getaddrinfo(ip, NULL, &hints, &res);
	if (err != 0)
		return(0);

	while (res)
	{
		inet_ntop (res->ai_family, res->ai_addr->sa_data, addrstr, sizeof(addrstr));

		if (res->ai_family != AF_INET)
			continue;

		ptr = &((struct sockaddr_in *) res->ai_addr)->sin_addr;
		inet_ntop (res->ai_family, ptr, addrstr, sizeof(addrstr));

		if (strcmp(addrstr, res->ai_canonname) != 0)
			json_object_add(jhost, "name", json_string_new(res->ai_canonname));

		res = res->ai_next;
	}
#endif
	return(0);

}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int broadcast_resp(char *ip, int port, BROADCAST_DATA *data, size_t len, void *user_data)
{
	unsigned char foscam_header[] = {0x4D, 0x4F, 0x5F, 0x49};
	JSON_VALUE *jhosts = (JSON_VALUE *) user_data;
	struct packet
	{
		struct ip ip;
		union
		{
			struct icmphdr icmp;
			struct tcphdr tcphdr;
		};
	} *pkt;
	int i;

	if (port == 0)
	{
		pkt = (struct packet *) data;

		switch (pkt->ip.ip_p)
		{
			case IPPROTO_ICMP:
				diagnostico_parse_icmp(jhosts, ip, data, len);
				break;

			case IPPROTO_TCP:
				diagnostico_parse_tcp(jhosts, ip, data, len);
				break;
		}

		return(0);
	}

	for (i = 0 ; i < sizeof(foscam_header) ; i++)
		if (foscam_header[i] != data->data[i])
			break;

	if (i >= sizeof(foscam_header))
		diagnostico_parse_siin_foscam(jhosts, ip, data, len);
	else if (data->data[0] == 'A')
		diagnostico_parse_tibbo(jhosts, ip, (char *) (data->data + 1), len);

/*
	printf("DATA: --> ");
	for (i = 0 ; i < len ; i++)
		printf("%02X ", data->data[i]);
	printf("<--\n\n");
*/

	return(0);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int diagnostico_check_hosts_iter(JSON_VALUE *jhost, void *user_data)
{
	JSON_VALUE *aux = json_object_find(jhost, "ip");
	char server[PATH_LEN];
	char *ip;
	struct
	{
		int (*func)(char *, int, char *, size_t);
		int port;
	} srvs[] = {
#ifndef WIN32
		{diagnostico_get_ssh_server, 22},
#endif
		{diagnostico_get_http_server, 80},
		{diagnostico_get_http_server, 8080},
		{NULL, 0}};
	int i, id;

	if (aux == NULL)
		return(0);

	ip = json_get_string(aux);

	aux = json_object_find(jhost, "id");
	if ((aux != NULL) && (json_get_int(aux) > DIAGNOSTICO_ID_UNKNOWN))
		return(0);

	for (i = 0 ; srvs[i].func != NULL ; i++)
	{
		server[0] = 0;
		if (!srvs[i].func(ip, srvs[i].port, server, PATH_LEN))
		{
			id = diagnostico_server_identification(jhost, server);
			if (id > 0)
				json_object_add(jhost, "id", json_integer_new(id));

			switch (id)
			{
				case DIAGNOSTICO_ID_IPCAM_FOSCAM:
				case DIAGNOSTICO_ID_IPCAM_ANDROID:
				case DIAGNOSTICO_ID_IPCAM_DLINK:
				case DIAGNOSTICO_ID_HENRY7x:
				case DIAGNOSTICO_ID_HENRY8x:
					json_object_add(jhost, 
						"status", 
						json_integer_new(DIAGNOSTICO_STATUS_UNREF));
					break;

				default:
					break;
			}

			diagnostico_add_port(jhost, srvs[i].port, server);
			if (srvs[i].port == 80)
				break;
		}
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_check_henry8x(JSON_VALUE *jhost, void *user_data)
{
	DIAGNOSTICO_ID *id = (DIAGNOSTICO_ID *) user_data;
	JSON_VALUE *jp1, *jp2;

	jp1 = diagnostico_find_port(jhost, 80);
	jp2 = diagnostico_find_port(jhost, 3000);
	if ((jp1 != NULL) && (jp2 != NULL))
	{
		json_object_add(jhost, "id", json_integer_new(DIAGNOSTICO_ID_HENRY8x));
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_UNREF));
		if (id != NULL)
			*id = DIAGNOSTICO_ID_HENRY8x;
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_check_hosts(JSON_VALUE *jhosts)
{
	int r;

	r = json_array_iter(jhosts, diagnostico_check_hosts_iter, NULL);

	json_array_iter(jhosts, diagnostico_check_henry8x, NULL);

	return(r);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int diagnostico_portscan_iter(JSON_VALUE *jhost, void *user_data)
{
	uint16_t *ports = (uint16_t *) user_data;
	JSON_VALUE *aux = json_object_find(jhost, "ip");
	char *host, port[PORT_LEN];
	int i, sock;

	if (aux == NULL)
		return(0);

	host = json_get_string(aux);

	for (i = 0 ; ports[i] != 0 ; i++)
	{
		if (diagnostico_find_port(jhost, ports[i]) != NULL)
			continue;

		snprintf(port, PORT_LEN, "%d", ports[i]);
		sock = openTCP(host, port, 2);
		if (sock < 1)
			continue;
        
		diagnostico_add_port(jhost, ports[i], "");

		if_closesocket(sock);
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_portscan(uint16_t ports[], JSON_VALUE *jhosts)
{
	json_array_iter(jhosts, diagnostico_portscan_iter, ports);

	return(1);
}
// //////////////////////////////////////////////////////////////////////



// //////////////////////////////////////////////////////////////////////
int diagnostico_add_readers_iter(JSON_VALUE *jreader, void *user_data)
{
	char *fields[] = {"nome","", "ip","", "porta","", "situacao","", NULL,NULL};
	JSON_VALUE *jhosts = (JSON_VALUE *) user_data;
	DIAGNOSTICO_ID id = DIAGNOSTICO_ID_UNKNOWN;
	JSON_VALUE *jhost, *jport, *aux;
	uint16_t ports[] = {80, 0, 0};
	unsigned char data[] = "SIIN";
	char *ip, *nome, *situacao;
	int porta;

	json_make_list_param(jreader, fields);

	nome = json_get_list_param(fields, "nome");
	ip = json_get_list_param(fields, "ip");
	porta = atoi(json_get_list_param(fields, "porta"));
	situacao = json_get_list_param(fields, "situacao");

	fprintf(stderr, "%s - %s:%d (%s)\n", nome, ip, porta, situacao);

	if (strstr(situacao, "ONLINE") != NULL)
	{
		jhost = diagnostico_add_host(jhosts, ip, nome, "module");
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_OK));
		goto diagnostico_add_readers_iter_fin;
	}

	ping_perform_timeout(ip, data, sizeof(data), broadcast_resp, jhosts, 1);

	jhost = diagnostico_find_host(jhosts, ip);
	if (jhost == NULL)
	{
		jhost = diagnostico_add_host(jhosts, ip, nome, "module");
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_UNREACHABLE));
		goto diagnostico_add_readers_iter_fin;
	}

	json_object_add(jhost, "name", json_string_new(nome));
	ports[1] = porta;
	diagnostico_portscan(ports, jhosts);

	diagnostico_check_henry8x(jhost, &id);
	if (id == DIAGNOSTICO_ID_HENRY8x)
	{
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_OK));
		goto diagnostico_add_readers_iter_fin;
	}
	jport = diagnostico_find_port(jhost, porta);
	if (jport == NULL)
	{
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_CONFLICT));
		goto diagnostico_add_readers_iter_fin;
	}

	aux = json_object_find(jhost, "id");
	if (aux == NULL)
		goto diagnostico_add_readers_iter_fin;

	id = atoi(json_get_string(aux));
	switch(id)
	{
		case DIAGNOSTICO_ID_HENRY7x:
		case DIAGNOSTICO_ID_HENRY8x:
			json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_OK));
			break;

		default:
			json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_CONFLICT));
			break;
	}

diagnostico_add_readers_iter_fin:
	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_get_readers_iter(JSON_VALUE *jfield, void *user_data)
{
	struct FIELD_PAIR
	{
		char *name;
		char **value;
	} *fields = (struct FIELD_PAIR *) user_data;
	char *list[] = {"name","", "value","", NULL,NULL};
	int i;

	json_make_list_param(jfield, list);

	for (i = 0 ; fields[i].name != NULL ; i++)
	{
		if (strcmp(fields[i].name, json_get_list_param(list, "name")) != 0)
			continue;

		*(fields[i].value) = json_get_list_param(list, "value");
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_get_readers(JSON_VALUE *jconf)
{
	JSON_VALUE *jreaders = NULL;
	char path[PATH_LEN];
	char *resp = NULL;
	char *webhost, *webport, *webpath, *readers;
	URL *url;
	struct
	{
		char *name;
		char **value;
	} fields[] = {
		{"WEBHOST", &webhost},
		{"WEBPORT", &webport},
		{"WEBPATH", &webpath},
		{"READERS", &readers},
		{NULL, NULL}
	};

	json_array_iter(jconf, diagnostico_get_readers_iter, fields);

	if (readers[0] == 0)
		snprintf(path, PATH_LEN, "%s/json_db.php?leitores=1", webpath);
	else
		snprintf(path, PATH_LEN, "%s/json_db.php?leitores=1&nros=%%5B%s%%5D", webpath, readers);
	url = url_make(webhost, webport, path);

	url_execute_get(url, (unsigned char **) &resp, NULL, NULL);
	url_free(url);

	if (resp == NULL)
		return(NULL);

	jreaders = json_parse_mem(resp);
	if_free(resp);

	return(jreaders);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_add_readers(JSON_VALUE *jhosts, JSON_VALUE *jhost, URL *url_conf)
{
	JSON_VALUE *jconf, *jdb, *jreaders;
	char *resp;

	url_execute_get(url_conf, (unsigned char **) &resp, NULL, NULL);
	if (resp == NULL)
	{
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_SIIN_UNREACHABLE));
		return(-1);
	}

	jconf = json_parse_mem(resp);
	if_free(resp);

	if (jconf == NULL)
		return(-2);

	jdb = diagnostico_get_readers(jconf);
	json_value_free(jconf);

	if (jdb == NULL)
		return(-3);

	jreaders = json_object_find(jdb, "leitores");
	if (jreaders == NULL)
		return(-4);

	json_array_iter(jreaders, diagnostico_add_readers_iter, jhosts);
	json_value_free(jdb);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_add_modules_iter(JSON_VALUE *jmodule, void *user_data)
{
	void **params = (void **) user_data;
	JSON_VALUE *jhosts = (JSON_VALUE *) params[0];
	JSON_VALUE *jhost = (JSON_VALUE *) params[1];
	char *ip = (char *) params[2];
	char *port = (char *) params[3];
	char *path = json_object_get_string(dic, "path");
	char webpath[PATH_LEN];
	URL *url;
	int i;
	struct
	{
		char *path;
		int (*func)(JSON_VALUE *jhosts, JSON_VALUE *jhost, URL *);
	} mods[] = {
		{"acesso", diagnostico_add_readers},
		{NULL, NULL}
	};

	if (path == NULL)
		goto diagnostico_add_modules_iter_fin;

	for (i = 0 ; mods[i].path != NULL ; i++)
	{
		if (strcmp(path, mods[i].path) != 0)
			continue;

		snprintf(webpath, PATH_LEN, "%s/config", path);
		url = url_make(ip, port, webpath);
		mods[i].func(jhosts, jhost, url);
		url_free(url);
	}

diagnostico_add_modules_iter_fin:

	dictionary_free(dic);
	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_add_modules(JSON_VALUE *jhosts, JSON_VALUE *jhost, char *ip, char *port)
{
	URL *url = url_make(ip, port, "modules");
	void *params[] = {jhosts, jhost, ip, port};
	JSON_VALUE *jmodules, *jport;
	char *resp;
	int n;

	n = url_execute_get(url, (unsigned char **) &resp, NULL, NULL);
	if (resp == NULL)
	{
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_MODULE_UNREACHABLE));
		goto diagnostico_add_modules_error;
	}

	jmodules = json_parse_mem(resp);
	if_free(resp);

	if (jmodules == NULL)
	{
		json_object_add(jhost, "status", json_integer_new(DIAGNOSTICO_STATUS_MODULE_FAIL));
		goto diagnostico_add_modules_error;
	}

	jport = diagnostico_find_port(jhost, atoi(port));
	json_object_add(jport, "modules", jmodules);

	json_array_iter(jmodules, diagnostico_add_modules_iter, params);

diagnostico_add_modules_error:
	url_free(url);

	return(n);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_siin_servers_check_iter(JSON_VALUE *jhost, void *user_data)
{
	JSON_VALUE *jhosts = (JSON_VALUE *) user_data;
	int id = atoi(json_object_get_string(dic, "id"));
	char *ip, *port;

	if (id != DIAGNOSTICO_ID_IFRACTAL)
		goto diagnostico_siin_servers_check_iter_error;

	ip = json_object_get_string(dic, "ip");
	port = diagnostico_get_port_from_id(jhost, DIAGNOSTICO_SERVICE_ID_SIIN);

	if ((ip == NULL) || (port == NULL))
		goto diagnostico_siin_servers_check_iter_error;

	diagnostico_add_modules(jhosts, jhost, ip, port);
	
diagnostico_siin_servers_check_iter_error:

	dictionary_free(dic);
	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_siin_servers_check(JSON_VALUE *jhosts)
{
	int r;

	r = json_array_iter(jhosts, diagnostico_siin_servers_check_iter, jhosts);

	return(r);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PRIVATE int diagnostico_sock_open()
{
	int sock;

#ifdef WIN32
	WSADATA WSData;
	if ( WSAStartup(MAKEWORD(2,2), &WSData) )
	{
		verbose(stderr, "ERROR: You need Winsock.\n");
		return(-1);
	}

	
	if ((sock = WSASocket(AF_INET, SOCK_DGRAM, IPPROTO_UDP, 0, 0, 0)) == SOCKET_ERROR)
	{
		fprintf(stderr, "ERROR creating socket (%d)\n", WSAGetLastError());
	  	return(-3);
	}

#else
	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
	{
		fprintf(stderr, "ERROR creating socket\n");
	  	return(-3);
	}
#endif

	return(sock);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
JSON_VALUE * diagnostico_perform_header(DIAGNOSTICO_CONFIG *conf)
{
	JSON_VALUE *jreport = json_object_new(1);
	unsigned char macs[MAX_IFACES][6];
	time_t now = time(NULL);
	struct sockaddr_in *dst;
	char buf[PATH_LEN];
	int ifs, i, sock;
	struct tm *dt;

	sock = diagnostico_sock_open();

#ifdef WIN32
	INTERFACE_INFO InterfaceList[MAX_IFACES];

	ifs = netraw_win_ifconfig(sock, InterfaceList, macs);
	for (i = 0; i < ifs ; i++)
	{
		dst = (struct sockaddr_in *) &(InterfaceList[i].iiAddress);

		if (((unsigned char *) &dst->sin_addr.s_addr)[0] == 127)
			continue;

		json_object_add(jreport, "ip", json_string_new(inet_ntoa(dst->sin_addr)));
	}

	// TODO
	//siin_mac(_INOUT char *buf, int buflen);
	json_object_add(jreport, "mac", json_string_new("---"));

#else
	struct ifconf ifconf;
	struct ifreq ifr[MAX_IFACES];
	struct ifreq *ifr_ptr;
	int n, sum, t;

	ifconf.ifc_buf = (char *) ifr;
	ifconf.ifc_len = sizeof(ifr);

	memset(macs, 0, sizeof(macs));

	ifs = netraw_ifconfig(sock, &ifconf, ifr, macs);

	for (i = 0, n = 0, ifr_ptr = ifr ; i < ifs ; i++, ifr_ptr = (struct ifreq *)(n + (char *) ifr_ptr))
	{
#ifdef __MACH__
		n = sizeof(struct ifreq) - sizeof(struct sockaddr) + MAX(sizeof(struct sockaddr), ifr_ptr->ifr_addr.sa_len);

		dst = (struct sockaddr_in *) &(ifr_ptr->ifr_addr);
#else
		dst = (struct sockaddr_in *) &ifr[i].ifr_addr;
#endif
		if (dst->sin_family != AF_INET)
			continue;

		if (((unsigned char *) &dst->sin_addr.s_addr)[0] == 127)
			continue;

		json_object_add(jreport, "ip", json_string_new(inet_ntoa(dst->sin_addr)));

		snprintf(buf, PATH_LEN, "%02X%02X%02X%02X%02X%02X", 
			macs[i][0],
			macs[i][1],
			macs[i][2],
			macs[i][3],
			macs[i][4],
			macs[i][5]);

		for (sum = 0, t = 0 ; t < 6 ; t++, sum += macs[i][t])
			;

		if (sum > 0)
			json_object_add(jreport, "mac", json_string_new(buf));
		else
			json_object_add(jreport, "mac", json_string_new("---"));
	}
#endif

	if_closesocket(sock);

#ifdef __linux__
	json_object_add(jreport, "os", json_string_new("GNU/Linux"));
#elif WIN32
	json_object_add(jreport, "os", json_string_new("Windows"));
#elif __MACH__
	json_object_add(jreport, "os", json_string_new("Macos"));
#endif

	snprintf(buf, PATH_LEN, "%ld", now);
	json_object_add(jreport, "unixtime", json_string_new(buf));

	dt = localtime(&now);
	snprintf(buf, PATH_LEN, "%02d/%02d/%d", dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900);
	json_object_add(jreport, "date", json_string_new(buf));

	snprintf(buf, PATH_LEN, "%02d:%02d:%02d", dt->tm_hour, dt->tm_min, dt->tm_sec);
	json_object_add(jreport, "time", json_string_new(buf));

	snprintf(buf, PATH_LEN, "iFractal SIIN - %d.%d.%d r%d", FWVER_A,FWVER_B,FWVER_C,FWREV);
	json_object_add(jreport, "version", json_string_new(buf));

	gethostname(buf, PATH_LEN);
	json_object_add(jreport, "hostname", json_string_new(buf));

	char flags[] = "----";
	for (i = 1 ; i < DIAGNOSTICO_CONFIG_FLAG_FIN ; i *= 2)
	{
		switch (conf->flags & i)
		{
			case DIAGNOSTICO_CONFIG_FLAG_SIIN: flags[0] = 's'; break;
			case DIAGNOSTICO_CONFIG_FLAG_ICMP: flags[1] = 'i'; break;
			case DIAGNOSTICO_CONFIG_FLAG_TIBBO: flags[2] = 't'; break;
			case DIAGNOSTICO_CONFIG_FLAG_PORT: flags[3] = 'p'; break;
			default: continue;
		}
	}

	json_object_add(jreport, "flags", json_string_new(flags));

	return(jreport);
}
// //////////////////////////////////////////////////////////////////////
_PRIVATE JSON_VALUE * diagnostico_perform_in(DIAGNOSTICO_CONFIG *conf)
{
	JSON_VALUE *jreport, *jhosts;
	BROADCAST_CONTEXT *ctx;
	time_t start, elapsed, last;
	int i, timeout = 10;
	char msg[PATH_LEN];

	jreport = diagnostico_perform_header(conf);
	jhosts = json_array_new(1);
	json_object_add(jreport, "hosts", jhosts);

	for (i = 1 ; i < DIAGNOSTICO_CONFIG_FLAG_FIN ; i *= 2)
	{
		switch (conf->flags & i)
		{
			case DIAGNOSTICO_CONFIG_FLAG_PORT:
				snprintf(msg, PATH_LEN, "Broadcast Portscan...");
#ifdef __linux__
				ctx = broadcast_portscan(conf->ports, broadcast_resp, jhosts);
#else
				diagnostico_portscan(conf->ports, jhosts);
				continue;
#endif
				break;

			case DIAGNOSTICO_CONFIG_FLAG_ICMP:
				snprintf(msg, PATH_LEN, "Broadcast ICMP...");
				ctx = broadcast_icmp(broadcast_resp, jhosts);
				break;

			case DIAGNOSTICO_CONFIG_FLAG_SIIN:
				snprintf(msg, PATH_LEN, "Broadcast SIIN...");
				ctx = broadcast_send_foscam_async(broadcast_resp, jhosts);
				break;

			case DIAGNOSTICO_CONFIG_FLAG_TIBBO:
				snprintf(msg, PATH_LEN, "Broadcast Tibbo...");
				ctx = broadcast_send_tibbo_async(broadcast_resp, jhosts);
				break;

			default:
				continue;
		}

		if (conf->event != NULL)
			conf->event(msg, conf->user_data);

		start = time(NULL);
		do
		{
			broadcast_events_pending(ctx);
        
			if_sleep(5);
        
			elapsed = time(NULL) - start;
			if (elapsed != last)
			{
				snprintf(msg, PATH_LEN, "Timout: %ld", timeout - elapsed);
				if (conf->event != NULL)
					conf->event(msg, conf->user_data);
				last = elapsed;
			}
		}
		while (elapsed < timeout);

		broadcast_free(ctx);
	}

	if (conf->event != NULL)
		conf->event("Server Check", conf->user_data);

	diagnostico_check_hosts(jhosts);

	if (conf->event != NULL)
		conf->event("Modulo Check", conf->user_data);

	diagnostico_siin_servers_check(jhosts);

	return(jreport);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_hosts_count_iter(JSON_VALUE *jhost, void *user_data)
{
	void **params = (void **) user_data;
	int *ids = (int *) params[0];
	int *status = (int *) params[1];
	int id, stat;

	id = atoi(json_object_get_string(dic, "id"));
	stat = atoi(json_object_get_string(dic, "status"));

	if ((id < 0) || (id >= DIAGNOSTICO_ID_LEN))
		return(0);

	if ((stat < 0) || (stat >= DIAGNOSTICO_STATUS_LEN))
		return(0);

	ids[id] += 1;
	status[stat] += 1;

	return(0);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * diagnostico_perform(DIAGNOSTICO_CONFIG *conf)
{
	JSON_VALUE *jreport = diagnostico_perform_in(conf);
	JSON_VALUE *jhosts = json_object_find(jreport, "hosts");
	JSON_VALUE *obj, *jhosts_qty, *jstatus_qty;
	int ids[DIAGNOSTICO_ID_LEN];
	int status[DIAGNOSTICO_STATUS_LEN];
	void *params[] = {ids, status};
	int i;

	jhosts_qty = json_array_new(1);
	json_object_add(jreport, "hosts_qty", jhosts_qty);

	jstatus_qty = json_array_new(1);
	json_object_add(jreport, "status_qty", jstatus_qty);

	memset(ids, 0, sizeof(ids));
	memset(status, 0, sizeof(status));

	json_array_iter(jhosts, diagnostico_hosts_count_iter, params);

	for (i = 0 ; i < DIAGNOSTICO_ID_LEN ; i++)
	{
		if (ids[i] > 0)
		{
			obj = json_object_new(1);
			json_object_add(obj, "id", json_integer_new(i));
			json_object_add(obj, "qty", json_integer_new(ids[i]));
			json_object_add(obj, "descript", json_string_new(diagnostico_get_ID_descript(i)));
			json_array_add(jhosts_qty, obj);
		}
	}

	if ((ids[DIAGNOSTICO_ID_IFRACTAL] == 0) && (ids[DIAGNOSTICO_ID_IFRACTAL_OLD] == 0))
	{
		i = DIAGNOSTICO_STATUS_MODULE_UNREACHABLE;
		obj = json_object_new(1);
		json_object_add(obj, "id", json_integer_new(i));
		json_object_add(obj, "qty", json_string_new("---"));
		json_object_add(obj, "descript", json_string_new(diagnostico_get_STATUS_descript(i)));
		json_array_add(jstatus_qty, obj);
	}

	for (i = 0 ; i < DIAGNOSTICO_STATUS_LEN ; i++)
	{
		if (status[i] > 0)
		{
			obj = json_object_new(1);
			json_object_add(obj, "id", json_integer_new(i));
			json_object_add(obj, "qty", json_integer_new(status[i]));
			json_object_add(obj, "descript", 
				json_string_new(diagnostico_get_STATUS_descript(i)));
			json_array_add(jstatus_qty, obj);
		}
	}

	return(jreport);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int diagnostico_show_port_iter(JSON_VALUE *jhost, void *user_data)
{
	FILE *fd = (FILE *) user_data;

	fprintf(fd, "\t%s (%s) - %s\n", 
		json_object_get_string(dic, "port"),
		json_object_get_string(dic, "default"),
		json_object_get_string(dic, "server"));

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int diagnostico_show_iter(JSON_VALUE *jhost, void *user_data)
{
	FILE *fd = (FILE *) user_data;
	int i = atoi(json_object_get_string(dic, "id"));
	JSON_VALUE *jports = json_object_find(jhost, "ports");

	fprintf(fd, "\n'%s' - %s (%s) - status: %s - macaddr: %s - %s\n", 
		json_object_get_string(dic, "name"),
		json_object_get_string(dic, "ip"),
		json_object_get_string(dic, "origem"),
		json_object_get_string(dic, "status"),
		json_object_get_string(dic, "mac"),
		diagnostico_get_ID_descript(i));

	dictionary_free(dic);

	if (jports != NULL)
		json_array_iter(jports, diagnostico_show_port_iter, fd);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
void diagnostico_show(JSON_VALUE *jreport, FILE *fd)
{
	JSON_VALUE *jhosts = json_object_find(jreport, "hosts");
	int ids[DIAGNOSTICO_ID_LEN];
	int status[DIAGNOSTICO_STATUS_LEN];
	void *params[] = {ids, status};
	int i, nhosts = 0;

	memset(ids, 0, sizeof(ids));
	memset(status, 0, sizeof(status));

	fprintf(fd, 
		"\n\n%s\nDiagnostico de Rede - %s %s (%s)\n\n" \
		"Hostname: %s  -  %s\n" \
		"IP: %s\n" \
		"MACADDR: %s\n\n", 
		json_object_get_string(dic, "version"),
		json_object_get_string(dic, "date"),
		json_object_get_string(dic, "time"),
		json_object_get_string(dic, "flags"),
		json_object_get_string(dic, "hostname"),
		json_object_get_string(dic, "os"),
		json_object_get_string(dic, "ip"),
		json_object_get_string(dic, "mac"));

	dictionary_free(dic);

	nhosts = json_array_iter(jhosts, diagnostico_hosts_count_iter, params);

	fprintf(fd, "Total de hosts localizados: %d\n", nhosts);
	for (i = 0 ; i < DIAGNOSTICO_ID_LEN ; i++)
		if (ids[i] > 0)
			fprintf(fd, "%2d - %s\n", ids[i], diagnostico_get_ID_descript(i));

	fprintf(fd, "\nStatus:\n");
	for (i = 0 ; i < DIAGNOSTICO_STATUS_LEN ; i++)
		if (status[i] > 0)
			fprintf(fd, "%2d - %s (%d)\n", status[i], diagnostico_get_STATUS_descript(i), i);

	json_array_iter(jhosts, diagnostico_show_iter, fd);

	fprintf(fd, "\n\n");
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
void diagnostico_html(JSON_VALUE *jreport, char *desktop_path)
{
	char path[PATH_LEN];
	char *html, *aux;
	JSON_VALUE *obj;
	FILE *fd;

	snprintf(path, PATH_LEN, "%s%cifractal_diagnostico.html", desktop_path, PATH_SEPARATOR);
	fd = fopen(path, "w");
	if (fd == NULL)
	{
		fprintf(stderr, "Falha ao tentar salvar: '%s'\n", path);
		return;
	}

	obj = json_object_new(1);

	aux = json_serialize(json_object_find(jreport, "hosts"));
	json_object_add(obj, "hosts_serialized", json_string_new(aux));
	if_free(aux);

	aux = json_serialize(json_object_find(jreport, "hosts_qty"));
	json_object_add(obj, "hosts_qty_serialized", json_string_new(aux));
	if_free(aux);

	aux = json_serialize(json_object_find(jreport, "status_qty"));
	json_object_add(obj, "status_qty_serialized", json_string_new(aux));
	if_free(aux);

	html = process_model(obj, jreport, DIAGNOSTICO_TEMPLATE_HTML);
	fprintf(fd, "%s", html);
	if_free(html);

	json_value_free(obj);

	fclose(fd);
}
// //////////////////////////////////////////////////////////////////////



#ifdef STANDALONE

#define DESCRIPT	"Faz varredura na rede local em busca de Modulos SIIN, IPCams, REPs, Controladoras de acesso.\n"


IF_GETOPT configs[] = {
#ifdef __linux__
	{0, 'p', IF_GETOPT_TYPE_STRING, "portscan", "", 0, "Portscan broadcast (half-open) - TCP syn."},
#else
	{0, 'p', IF_GETOPT_TYPE_STRING, "portscan", "", 0, "Portscan (full-open)."},
#endif
	{0, 'i', IF_GETOPT_TYPE_NONE, "icmp", "", 0, "ICMP broadcast."},
	{0, 's', IF_GETOPT_TYPE_NONE, "siin", "", 0, "SIIN/Foscam UDP broadcast."},
	{0, 't', IF_GETOPT_TYPE_NONE, "tibbo", "", 0, "Tibbo UDP broadcast."},
	{0, 0, 0, 0, 0, 0, 0}
};


// //////////////////////////////////////////////////////////////////////
_CALLBACK int event_callback(char *msg, void *user_data)
{
	FILE *fd = (FILE *) user_data;

	fprintf(fd, "%s\n", msg);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	char desktop_path[PATH_LEN] = "./";
	DIAGNOSTICO_CONFIG conf;
	JSON_VALUE *jreport;
	FILE *fd = stdout;
	int r, i;
	char *p;

	r = if_getopt(configs, argc, argv);

#ifdef WIN32

	if (SHGetFolderPath(NULL, CSIDL_DESKTOPDIRECTORY, NULL, 0, desktop_path) != S_OK)
		desktop_path[0] = 0;

	if (argc < 2)
	{
		if_getopt_check(configs, "icmp");
		if_getopt_check(configs, "tibbo");
		if_getopt_check(configs, "siin");
	}
	else
#endif

	if (	(r < 0) || 
		(!if_getopt_isChecked(configs, "icmp") 
		&& !if_getopt_isChecked(configs, "tibbo")
		&& !if_getopt_isChecked(configs, "siin"))
		)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s [OPCOES]\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\tshell$ %s -sit -p 5800,6000,4500\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

	memset(&conf, 0, sizeof(conf));

	if (if_getopt_isChecked(configs, "icmp"))
		conf.flags |= DIAGNOSTICO_CONFIG_FLAG_ICMP;

	if (if_getopt_isChecked(configs, "siin"))
		conf.flags |= DIAGNOSTICO_CONFIG_FLAG_SIIN;

	if (if_getopt_isChecked(configs, "tibbo"))
		conf.flags |= DIAGNOSTICO_CONFIG_FLAG_TIBBO;

	if (if_getopt_isChecked(configs, "portscan"))
	{
		conf.flags |= DIAGNOSTICO_CONFIG_FLAG_PORT;

		p = if_getopt_getValue(configs, "portscan");
		for (i = 0 ; (*p != 0) && (i < PATH_LEN) ; i++)
		{
			conf.ports[i] = atoi(p);
			while ((*p != 0) && (*p != ','))
				p++;
		}
	}

	conf.event = event_callback;
	conf.user_data = stderr;

	jreport = diagnostico_perform(&conf);

	diagnostico_show(jreport, fd);
	diagnostico_html(jreport, desktop_path);

	json_value_free(jreport);

	fclose(fd);

#ifdef WIN32
	getchar();
#endif

	return(0);
}
// //////////////////////////////////////////////////////////////////////
#endif

