#include <ifractal.h>
#include <dimep.h>
#include <ifdevice.h>


// ///////////////////////////////////////////////////////// //
_PRIVATE void dimep_reconnect(IFDEVICE4J *dev)
{
	dimep_close(dev);
	if_sleep(10*1000);
	dimep_connect(dev);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PUBLIC void dimep_checksum(_INOUT DIMEP_PACK *pack)
{
	int i;

	if ((pack == NULL) || (pack->datalen < 0)) 
		return;

	pack->header[DIMEP_HEADER_MSB_SIZE] = (pack->datalen & 0xFF00) >> 8;
	pack->header[DIMEP_HEADER_LSB_SIZE] = pack->datalen & 0xFF;

	pack->checksum = 0;

	for (i = 1 ; i < DIMEP_HEADER_LEN ; i++)
		pack->checksum ^= pack->header[i];

	for (i = 0 ; i < pack->datalen ; i++)
		pack->checksum ^= pack->data[i];
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PUBLIC int dimep_connect(IFDEVICE4J *dev)
{
	if (dev == NULL)
		return(-1);

	dev->sock = openTCP(dev->host, dev->port, 20);
	((DIMEP *) dev)->sequence = 0;

	return(dev->sock);
}
// ///////////////////////////////////////////////////////// //
_PUBLIC int dimep_close(IFDEVICE4J *dev)
{
	if (dev == NULL)
		return(-1);

	if (dev->sock > 0)
		return(if_closesocket(dev->sock));

	return(0);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PUBLIC void dimep_pack_free(_INOUT DIMEP_PACK *pack)
{
	if (pack == NULL)
		return;

	if (pack->data != NULL)
		if_free(pack->data);

	if_free(pack);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PRIVATE int dimep_send_with_noise(_IN IFDEVICE4J *dev, _INOUT DIMEP_PACK *pack)
{
	unsigned char endbyte = DIMEP_END;
	struct DIMEP_BLOCK
	{
		size_t length;
		unsigned char *source;
	} blocks[] = {
		{DIMEP_HEADER_LEN, pack->header},
		{pack->datalen, pack->data},
		{1, &(pack->checksum)},
		{1, &endbyte},
		{0, NULL}
	};
	unsigned char *buf;
	int l, c, i, k, n;
	char *log;

	n = sizeof(blocks) / sizeof(struct DIMEP_BLOCK);

	for (l = 0, c = 0 ; c < n ; c++)
		l += blocks[c].length;

	buf = if_malloc(2 * l);

	for (c = 0, k = 0 ; c < n ; c++)
	{
		for (i = 0 ; i < blocks[c].length ; i++, k++)
		{
			buf[k] = blocks[c].source[i];
			if (buf[k] == 0x10)
			{
				k++;
				buf[k] = 0x40;
			}
		}
	}

	if (send_bytes(dev->sock, buf, k, 20) < k)
		return(-2);

	log = siin_hexlog((unsigned char *) buf, k);
	verboseDEBUG(&(dev->log), "(%d) iFractal --> leitor   bytes: %d\n%s\n\n", dev->nro, k, log);
	if_free(log);

	fflush(stdout);

	if_free(buf);

	return(k);
}
// ///////////////////////////////////////////////////////// //
_PUBLIC int dimep_send(
	_IN IFDEVICE4J *dev, 
	_IN unsigned char cmd, 
	_IN unsigned char *data, 
	_IN size_t datalen)
{
	DIMEP_PACK pack;
	int r;

	if (dev == NULL)
		return(-1);

	pack.header[0] = DIMEP_START1;
	pack.header[1] = DIMEP_START2;
	pack.header[2] = cmd;
	pack.header[3] = ((DIMEP *) dev)->sequence;

	if ((data == NULL) || (datalen < 1))
	{
		pack.data = NULL;
		pack.datalen = 0;
	}
	else
	{
		pack.data = data;
		pack.datalen = datalen;
	}

	dimep_checksum(&pack);

	r = dimep_send_with_noise(dev, &pack);

	return(r);
}
// ///////////////////////////////////////////////////////// //
_PUBLIC int dimep_send_ack(_IN IFDEVICE4J *dev)
{
	int r;

	if_sleep(500);
	((DIMEP *) dev)->sequence += 1;
	r = dimep_send(dev, DIMEP_CMD_ACK, NULL, 0);
	((DIMEP *) dev)->sequence += 1;

	return(r);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PRIVATE int dimep_decode_block(
	_IN IFDEVICE4J *dev, 
	_INOUT DIMEP_PACK *pack, 
	_IN unsigned char *buf, 
	int len)
{
	char *log;
	int i, j;

	if (len < DIMEP_HEADER_LEN)
		return(-1);

	pack->blocklen = len;

	for (i = 0 ; i < len ; i++)
	{
		if (buf[i] != 0x10)
			continue;

		for (j = i + 1 ; j < len ; j++)
			buf[j] = buf[j + 1];

		buf[len - 1] = 0;
	}

	memcpy(pack->header, buf, DIMEP_HEADER_LEN);

	pack->datalen = pack->header[DIMEP_HEADER_MSB_SIZE] << 8;
	pack->datalen += pack->header[DIMEP_HEADER_LSB_SIZE];

	if ((pack->datalen > 0) && (pack->datalen < pack->blocklen))
	{
		pack->data = if_malloc(pack->blocklen);
		memcpy(pack->data, buf + DIMEP_HEADER_LEN, pack->blocklen - DIMEP_HEADER_LEN);

		log = siin_hexlog((unsigned char *) pack->data, pack->datalen);
		verboseDEBUG(&(dev->log), "(%d) PACK - bytes: %d\n%s\n\n", dev->nro, pack->datalen, log);
		if_free(log);
	}
	else
		pack->data = NULL;

	return(len);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE DIMEP_PACK * dimep_recv(_IN IFDEVICE4J *dev)
{
	DIMEP_PACK *pack = NULL;
	unsigned char *buf;
	char *log;
	int n;

	if (dev == NULL)
		return(NULL);

	buf = if_malloc(BUFFER_LEN);

	if  ((n = read_timeout(dev->sock, buf, BUFFER_LEN, 30)) < DIMEP_HEADER_LEN)
	{
		verboseWARN(&(dev->log), "(%d) Erro ao tentar ler pacote: %d\n", dev->nro, n);
		goto dimep_recv_err;
	}

	log = siin_hexlog((unsigned char *) buf, n);
	verboseDEBUG(&(dev->log), "(%d) iFractal <-- leitor   bytes: %d\n%s\n\n", dev->nro, n, log);
	if_free(log);

	pack = if_malloc(sizeof(DIMEP_PACK));
	if (dimep_decode_block(dev, pack, buf, n) < 0)
	{
		if_free(pack);
		pack = NULL;
	}

dimep_recv_err:
	if_free(buf);

	return(pack);
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
_PRIVATE int dimep_getConfig_fillBuffer(
	_IN unsigned char *data,
	_IN int len,
	_INOUT char buf[])
{
	char *p;
	int i;

	for (p = buf, i = 0 ; i < len ; i++)
		p += snprintf(p, ((p - buf) - PATH_LEN - 1), "%02X", data[i]);

	return(i);
}
// ///////////////////////////////////////////////////////// //
int dimep_getConfig(_IN IFDEVICE4J *dev, JSON_VALUE *info)
{
	struct
	{
		char *param;
		unsigned char id[2];
	} params[] = {
		{"now", {0x10, 0x1}},
		{"Horario Verao", {0x10, 0x0}},
		{"MACAddress", {0x60, 0x30}},
		{"NSR", {0x50, 0x70}},
		{"Serial", {0x60, 0x0}},
		{"Firmware", {0x60, 0x20}},
		{"Total de Registros", {0x50, 0x71}},
		{"Capacidade de Biometrias", {0x50, 0x41}},
		{"Biometrias cadastradas", {0x50, 0x50}},
		{"Ocupacao MRP", {0x50, 0x60}},
		{NULL, {0, 0}}
	};
	char buf[PATH_LEN];
	DIMEP_PACK *pack;
	int i, n;

	if ((n = dimep_send(dev, DIMEP_CMD_STATUS, NULL, 0)) < 0)
		return(n);

	pack = dimep_recv(dev);
	if (pack == NULL)
		return(-10);

	if ((n = dimep_send_ack(dev)) < 0)
		return(n);

	dimep_pack_free(pack);

	pack = dimep_recv(dev);
	if (pack == NULL)
		return(-11);

	if (pack->blocklen <= 8)
		return(-12);

	for (i = 0 ; i < pack->blocklen ; i += 3 + pack->data[i + 2])
	{
		for (n = 0 ; params[n].param != NULL ; n++)
		{
			if (pack->data[i] != params[n].id[0])
				continue;

			if (pack->data[i + 1] != params[n].id[1])
				continue;

			dimep_getConfig_fillBuffer(pack->data + i + 3, pack->data[i + 2], buf);

			json_object_add(info, params[n].param, json_string_new(buf));
		}
	}

	dimep_pack_free(pack);

	return(0);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
int dimep_setTime_in(_IN IFDEVICE4J *dev, _IN char timediff)
{
	unsigned char *data = if_malloc(9);
	DIMEP_PACK *pack;
	struct tm *dt;
	time_t now;
	int n, i;

	now = time(NULL);
	now += (3600 * timediff);
	dt = localtime(&now);

	i = 0;
	data[i] = (dt->tm_mday / 10) << 4;
	data[i++] += dt->tm_mday % 10;

	data[i] = ((dt->tm_mon + 1) / 10) << 4;
	data[i++] += (dt->tm_mon + 1) % 10;

	data[i++] = 0x20;

	data[i] = ((dt->tm_year - 100) / 10) << 4;
	data[i++] += (dt->tm_year - 100) % 10;

	data[i] = (dt->tm_hour / 10) << 4;
	data[i++] += dt->tm_hour % 10;

	data[i] = (dt->tm_min / 10) << 4;
	data[i++] += dt->tm_min % 10;

	data[i] = (dt->tm_sec / 10) << 4;
	data[i++] += dt->tm_sec % 10;

	if ((n = dimep_send(dev, DIMEP_CMD_DATAHORA, data, 7)) < 0)
		return(n);

	pack = dimep_recv(dev);
	dimep_pack_free(pack);

	if ((n = dimep_send_ack(dev)) < 0)
		return(n);

	pack = dimep_recv(dev);
	dimep_pack_free(pack);

	return(n);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PRIVATE int dimep_coleta_in(_IN IFDEVICE4J *dev, _IN unsigned char cmd, _INOUT JSON_VALUE *joffs)
{

	unsigned char aux[] = {0, 2};
	unsigned char *data;
	char pis[20], datahora[40], afd[PATH_LEN];
	JSON_VALUE *jevent;
	DIMEP_PACK *pack;
	int n, i, k, r, nsr;
	char *p;

	if (cmd == DIMEP_CMD_COLETA_PRIMEIRA)
		n = dimep_send(dev, cmd, aux, 2);
	else
		n = dimep_send(dev, cmd, NULL, 0);

	pack = dimep_recv(dev);
	dimep_pack_free(pack);

	if ((n = dimep_send_ack(dev)) < 0)
	{
		r = -2;
		goto dimep_coleta_in_end;
	}

	pack = dimep_recv(dev);
	if ((pack == NULL) || (pack->datalen < 1))
	{
		r = 0;
		dimep_pack_free(pack);
		goto dimep_coleta_in_end;
	}

	jevent = json_object_new(1);
	json_array_add(joffs, jevent);

	// Info nao deduzida
	k = 3; 
	data = pack->data;

	// NSR
	for (p = afd, i = 0 ; i < 5 ; i++, k++)
		p += snprintf(p, PATH_LEN, "%02X", data[k]);
	nsr = atoi(afd);
	json_object_add(jevent, "nsr", json_integer_new(nsr));

	// Data/hora
	snprintf(datahora, sizeof(datahora), "%02X%02X-%02X-%02X %02X:%02X:00", 
		data[k + 2], data[k + 3], data[k + 1], data[k + 0], data[k + 4], data[k + 5]);
	json_object_add(jevent, "datahora", json_string_new(datahora));

	// PIS
	for (k += 6, p = pis, i = 0 ; i < 6 ; i++, k++)
		p += snprintf(p, sizeof(pis), "%02X", data[k]);
	json_object_add(jevent, "pis", json_string_new(pis));

	// AFD
	snprintf(afd, PATH_LEN, "%09d3%c%c%c%c%c%c%c%c%c%c%c%c%s", nsr, 
		datahora[8], datahora[9],
		datahora[5], datahora[6],
		datahora[0], datahora[1], datahora[2], datahora[3],
		datahora[11], datahora[12],
		datahora[14], datahora[15], pis);
	json_object_add(jevent, "afd", json_string_new(afd));

	r = 1;

dimep_coleta_in_end:
	return(r);
}
// ///////////////////////////////////////////////////////// //
int dimep_coleta(_IN IFDEVICE4J *dev, JSON_VALUE *joffs)
{
	int r;

	r = dimep_coleta_in(dev, DIMEP_CMD_COLETA_PRIMEIRA, joffs);
	while (r > 0)
	{
		verboseDEBUG(&(dev->log), "OFFs: %d\n", json_array_length(joffs));
		r = dimep_coleta_in(dev, DIMEP_CMD_COLETA_PROXIMO, joffs);
	}

	r = json_array_length(joffs);

	return(r);
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
_PRIVATE int dimep_str2bcd(_IN char *str, _INOUT unsigned char buf[], int len)
{
	int i, k;

	if ((str == NULL) || (buf == NULL) || (len < 1))
		return(-1);

	for (i = 0 ; str[i] != 0 ; i++)
	{
		for (k = 0 ; k < (len - 1) ; k++)
		{
			buf[k] <<= 4;
			buf[k] |= (buf[k + 1] >> 4);
		}

		buf[k] <<= 4;
		buf[len - 1] |= str[i] - '0';
	}

	return(i);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_CALLBACK int dimep_load_list_iter(_IN JSON_VALUE *jpessoa, void *user_data)
{
	void **pars = (void **) user_data;
	IFDEVICE4J *dev = (IFDEVICE4J *) pars[0];
	JSON_VALUE *res = (JSON_VALUE *) pars[1];
	DIMEP_SEND_NAME lista_nome = {
		{1, 1, 0}, {0, 1},
		{0, 0, 0, 0, 0, 0},
		"",
		{0, 0, 0}
	};
	DIMEP_SEND_MATRICULA lista_matricula = {
		{1, 1, 0}, {0x20},
		{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		{0, 0, 0, 0, 0, 0}
	};
	unsigned char oper = DIMEP_CMD_OPER_INSERT;
	char *nome, *pis, *cracha, *senha;
	DIMEP_PACK *pack;
	int n, i;

	nome = json_object_get_string(jpessoa, "nome");
	pis = json_object_get_string(jpessoa, "pis");
	cracha = json_object_get_string(jpessoa, "cracha");
	senha = json_object_get_string(jpessoa, "senha");

	if (atoi(json_object_get_string(jpessoa, "tipo_operacao")) == 3)
		oper = DIMEP_CMD_OPER_DELETE;

	lista_nome.config[DIMEP_CMD_OPER_BYTE] = oper;
	lista_matricula.config[DIMEP_CMD_OPER_BYTE] = oper;

	memset(lista_nome.nome, ' ', sizeof(lista_nome.nome));
	n = snprintf(lista_nome.nome, sizeof(lista_nome.nome), "%s", nome);
	lista_nome.nome[n] = ' ';

	dimep_str2bcd(senha, lista_nome.senha, sizeof(lista_nome.senha));
	dimep_str2bcd(pis, lista_nome.pis, sizeof(lista_nome.pis));
	memcpy(lista_matricula.pis, lista_nome.pis, sizeof(lista_nome.pis));

	dimep_str2bcd(cracha, lista_matricula.cracha, sizeof(lista_matricula.cracha));

	for (i = 2 ; i > 0 ; i--)
	{
		n = dimep_send(dev, DIMEP_CMD_ENVIA_NOME, 
			(unsigned char *) &lista_nome, sizeof(DIMEP_SEND_NAME));
        
		pack = dimep_recv(dev);
		if (pack == NULL)
		{
			verboseFATAL(&(dev->log), "Falha ao tentar enviar lista... (1)\n");
			dimep_reconnect(dev);
			continue;
		}
		dimep_pack_free(pack);
        
		((DIMEP *) dev)->sequence += 1;
        
		n = dimep_send(dev, DIMEP_CMD_ENVIA_MATRICULA, 
			(unsigned char *) &lista_matricula, sizeof(DIMEP_SEND_MATRICULA));
        
		pack = dimep_recv(dev);
		if (pack == NULL)
		{
			verboseFATAL(&(dev->log), "(%d) Falha ao tentar enviar lista... (2)\n", dev->nro);
			dimep_reconnect(dev);
			continue;
		}
		dimep_pack_free(pack);
        
		if ((n = dimep_send_ack(dev)) < 0)
			return(n);
        
		pack = dimep_recv(dev);
		if ((pack == NULL) || (pack->datalen > 0))
		{
			verboseFATAL(&(dev->log), "(%d) Inconsistencia na resposta ao enviar lista...\n", dev->nro);
			dimep_reconnect(dev);
			continue;
		}

		dimep_pack_free(pack);
        
		JSON_VALUE *reg = json_object_new(1);
		json_array_add(res, reg);
		json_object_add(reg, "pis", json_string_new(pis));
		json_object_add(reg, "cod_error", json_integer_new(0));

		return(0);
	}

	return(-4);
}
// ///////////////////////////////////////////////////////// //


// Implementacao /////////////////////////////////////////// //
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_getInfo(IFDEVICE4J *dev, intptr_t sock)
{
	JSON_VALUE *info = NULL;

	int r = dimep_connect(dev);
	if (r < 1)
		return(NULL);

	info = json_object_new(1);
	dimep_getConfig(dev, info);

	dimep_close(dev);

	return(info);
}
// ///////////////////////////////////////////////////////// //
time_t dimep_getTime(IFDEVICE4J *dev, intptr_t sock)
{
	JSON_VALUE *info = dimep_getInfo(dev, sock);
	struct tm dt;
	int i = 0;

	if (info == NULL)
		return(0);

	char *strnow = json_object_get_string(info, "now");

	dt.tm_mday = (strnow[i++] - '0') * 10;
	dt.tm_mday += (strnow[i++] - '0');
	dt.tm_mon = (strnow[i++] - '0') * 10;
	dt.tm_mon += (strnow[i++] - '0') - 1;
	i += 2;
	dt.tm_year = (strnow[i++] - '0') * 10;
	dt.tm_year += (strnow[i++] - '0') + 100;
	dt.tm_hour = (strnow[i++] - '0') * 10;
	dt.tm_hour += (strnow[i++] - '0');
	dt.tm_min = (strnow[i++] - '0') * 10;
	dt.tm_min += (strnow[i++] - '0');
	dt.tm_sec = (strnow[i++] - '0') * 10;
	dt.tm_sec += (strnow[i++] - '0');

	json_value_free(info);

	time_t now = mktime(&dt);

	return(now);
}
// ///////////////////////////////////////////////////////// //
intptr_t dimep_setTime(IFDEVICE4J *dev, intptr_t diff)
{
	int r = dimep_connect(dev);
	if (r < 1)
		return(0);

	dimep_setTime_in(dev, diff);

	dimep_close(dev);

	return(1);
}
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_sendUsers(IFDEVICE4J *dev, JSON_VALUE *users)
{
	JSON_VALUE *res = NULL;

	int r = dimep_connect(dev);
	if (r < 1)
		return(0);

	res = json_array_new(1);
	void *pars[] = {dev, res};
	json_array_iter(users, dimep_load_list_iter, pars);

	dimep_close(dev);

	return(res);
}
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_getEvents(IFDEVICE4J *dev, intptr_t nsr)
{
	JSON_VALUE *joffs = NULL;
	int r = dimep_connect(dev);
	if (r < 1)
		return(NULL);

	joffs = json_array_new(1);
	dimep_coleta(dev, joffs);

	dimep_close(dev);

	return(joffs);
}
// ///////////////////////////////////////////////////////// //
void dimep_free(IFDEVICE4J *dev)
{
}
// ///////////////////////////////////////////////////////// //
int dimep_online(IFDEVICE4J *dev, IFDEVICE4J_online_callback callback, void *ctx)
{
	return(10);
}
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_getUsers(IFDEVICE4J *dev, intptr_t none)
{
	return(NULL);
}
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_sendBio(IFDEVICE4J *dev, JSON_VALUE *users)
{
	return(NULL);
}
// ///////////////////////////////////////////////////////// //
JSON_VALUE * dimep_getBio(IFDEVICE4J *dev, JSON_VALUE *users)
{
	return(NULL);
}
// ///////////////////////////////////////////////////////// //
void dimep_init(IFDEVICE *dev)
{
	dev->getInfo = dimep_getInfo;
	dev->getTime = dimep_getTime;
	dev->setTime = dimep_setTime;
	dev->sendUsers = dimep_sendUsers;
	dev->getUsers = dimep_getUsers;
	dev->sendBio = dimep_sendBio;
	dev->getBio = dimep_getBio;
	dev->getEvents = dimep_getEvents;
	dev->free = dimep_free;
	dev->online = dimep_online;
}
// ///////////////////////////////////////////////////////// //
// Fim implementacao /////////////////////////////////////// //


