#include <ifractal.h>
#include <ifdevice.h>

#include <dotnet.h>

char dllname[256];


#ifdef WIN32

extern ICLRRuntimeHost *runtime;

// ///////////////////////////////////////////////////////////////////// //
int dotnet_load_mono(char *lib, char *ns, char *klass)
{
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

#else

#include <mono/jit/jit.h>
#include <mono/metadata/assembly.h>

pthread_mutex_t mono_mutex = PTHREAD_MUTEX_INITIALIZER;

#define S_OK	0
MonoDomain *runtime = NULL;
MonoAssembly *assembly = NULL;
MonoImage *monoimg = NULL;
MonoClass *klass = NULL;


#define FUNCTION_NAME(name) #name

typedef struct
{
	void * (*thread_attach)(void *);
	void (*thread_detach)(void *);
	void * (*get_root_domain)();
	MonoDomain * (*jit_init_version)(const char *, const char *);
	MonoAssembly * (*domain_assembly_open)(MonoDomain *, const char *);
	MonoImage * (*assembly_get_image)(MonoAssembly *);
	MonoClass * (*class_from_name)(MonoImage *, char *, char *);
	MonoMethod * (*class_get_methods)(MonoClass *, void *);
	char * (*method_get_name)(MonoMethod *);
	void * (*string_new)(MonoDomain *, char *);
	MonoObject * (*runtime_invoke)(MonoMethod *, void *, void *, void *);
	void * (*object_unbox)(void *);
} IFACE_MONO;

char *mono_names[] = {
	"mono_thread_attach",
	"mono_thread_detach",
	"mono_get_root_domain",
	"mono_jit_init_version",
	"mono_domain_assembly_open",
	"mono_assembly_get_image",
	"mono_class_from_name",
	"mono_class_get_methods",
	"mono_method_get_name",
	"mono_string_new",
	"mono_runtime_invoke",
	"mono_object_unbox"
};

IFACE_MONO imono;

typedef void (*GENERIC_FUNC)();

static_assert (sizeof(imono) == sizeof(mono_names), "IFACE_MONO e mono_names devem ter o mesmo tamanho.");


// ///////////////////////////////////////////////////////////////////// //
int dotnet_load_mono(char *lib, char *ns, char *cl)
{
	void *handler;

	if ((lib == NULL) || (lib[0] == 0))
	{
		verbose(stderr, "mono_library nao definido.\n");
		return(1);
	}

	if ((handler = dlopen(lib, RTLD_LAZY)) == NULL)
	{
		verbose(stderr, "%s\n", dlerror());
		fflush(stderr);
		return(2);
	}

	GENERIC_FUNC *funcs = (GENERIC_FUNC *) &imono;

	void *func; 
	for (int i = 0 ; i < (sizeof(mono_names) / sizeof(char *)) ; i++)
	{
		func = (void *) dlsym(handler, mono_names[i]);

		if (func == NULL)
		{
			verbose(stderr, "Metodo '%s' nao localizado em '%s'.\n", mono_names[i], lib);
			return(3);
		}

		funcs[i] = (GENERIC_FUNC) func;
	}

	runtime = imono.jit_init_version(dllname, "v4.0");
	if (runtime == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar MONO.\n");
		return(4);
	}

	assembly = imono.domain_assembly_open(runtime, dllname);
	if (assembly == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar MONO assembly: '%s'.\n", dllname);
		return(5);
	}

	monoimg = imono.assembly_get_image(assembly);
	if (monoimg == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar MONO (image).\n");
		return(6);
	}

	klass = imono.class_from_name(monoimg, ns, cl);
	if (klass == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar MONO classe: %s.%s\n", ns, cl);
		return(7);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int callDefaultDotNetMethod(MonoDomain *runtimeHost, char *dll, char *namespace_class, char *meth_name, char *arg, int32_t *returnValue)
{
	char buf[PATH_LEN];
	char *tk[4];
	int n, r = 0;

	if ((runtime == NULL) || (assembly == NULL) || (monoimg == NULL) || (klass == NULL))
		return(0);

	// TODO - nome completo da classe pode ter varios caracteres '.'

	snprintf(buf, PATH_LEN, "%s", namespace_class);
	n = tokenizer('.', buf, tk, sizeof(tk) / sizeof(char *));
	if (n < 2)
		return(1);

	MonoMethod *meth = NULL, *m;
	void *iter = NULL;
	while ((m = imono.class_get_methods(klass, &iter)))
		if (strcmp (imono.method_get_name(m), meth_name) == 0)
			meth = m;

	if (meth == NULL)
	{
		r = 3;
		goto callDefaultDotNetMethod_end;
	}

	void *args[] = {imono.string_new(runtime, arg)};
	MonoObject *result = imono.runtime_invoke(meth, NULL, args, NULL);
	*returnValue = *(int32_t *) imono.object_unbox(result);

callDefaultDotNetMethod_end:

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

#endif


// ///////////////////////////////////////////////////////////////////// //
char * dotnet_address_recovery(char *ref, int32_t ret)
{
	uint32_t lower = (uint32_t) ret;
	uint64_t upper;
	char *ptr = NULL;

	if (ret == 0)
		return(NULL);

#ifdef WIN32

	upper = (uint64_t) ref & 0xFFFFFFFF00000000;

#else

	MonoMethod *meth = NULL, *m;
	void *iter = NULL;
	while ((m = imono.class_get_methods(klass, &iter)))
		if (strcmp (imono.method_get_name(m), "GetPtr") == 0)
			meth = m;

	if (meth == NULL)
	{
		upper = lower = 0;
		goto dotnet_address_recovery_end;
	}

	void *args[] = {&ret};
	MonoObject *result = imono.runtime_invoke(meth, NULL, args, NULL);
	upper = *(int64_t *) imono.object_unbox(result);
	upper &= 0xFFFFFFFF00000000;

dotnet_address_recovery_end:

#endif

	ptr = (char *)(upper + (uint64_t) lower);

	return(ptr);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void * dotnet_open_vm()
{
	void *mono_th = NULL;

#ifdef WIN32

	if (runtime == NULL)
		runtime = loadDotNetInstance();

#else

	if (imono.jit_init_version == NULL)
		return(NULL);

	pthread_mutex_lock( &mono_mutex );
	// TODO - erro fatal nesse ponto
	mono_th = imono.thread_attach(runtime);
	pthread_mutex_unlock( &mono_mutex );

#endif

	return(mono_th);
}
// ///////////////////////////////////////////////////////////////////// //
void dotnet_close_vm(void *th)
{
#ifndef WIN32
	if (th == NULL)
		imono.thread_detach(th);
#endif
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
size_t dotnet_setDllName(char *name)
{
	int32_t len = 0;

	if (name != NULL)
	{
		len = snprintf(dllname, sizeof(dllname), "%s", name);
		return(len);
	}

#ifdef WIN32
	len = GetCurrentDirectory(sizeof(dllname), dllname);
	len += snprintf(dllname + len, sizeof(dllname) - len, "\\bin\\siin.Net.exe");
#else
	strncpy(dllname, "./bin/siin.MONO.exe", sizeof(dllname));
	len = strlen(dllname);
#endif
	verbose(stderr, "Load '%s'\n", dllname);

	return(len);
}
// ///////////////////////////////////////////////////////////////////// //


// Implementacao interface Java //////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //
char * dotnet_getModels()
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	char *strinfo = if_strdup(siin_version());
	int32_t returnValue = 0;
	char *models = NULL;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetModels", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	models = dotnet_address_recovery(strinfo, returnValue);

	if_free(strinfo);

	dotnet_close_vm(th);

	return(models);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_getInfo4J(IFDEVICE4J *dev, long sock)
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	JSON_VALUE *jinfo = NULL;
	char *strinfo = json_serialize(dev->config);
	int32_t returnValue = 0;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetInfo", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);
	if_free(strinfo);

	if(ptr != NULL)
		jinfo = json_parse_mem(ptr);

	dotnet_close_vm(th);

	return(jinfo);
}
// ///////////////////////////////////////////////////////////////////// //
time_t dotnet_getTime4J(IFDEVICE4J *dev, long sock)
{
	if (runtime == NULL)
		return(0);

	void *th = dotnet_open_vm();

	char *strinfo = json_serialize(dev->config);
	int32_t returnValue = 0;
	time_t dt = 0;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetTime", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	dt = (time_t) returnValue;

	if_free(strinfo);

	dotnet_close_vm(th);

	return(dt);
}
// ///////////////////////////////////////////////////////////////////// //
intptr_t dotnet_setTime4J(IFDEVICE4J *dev, intptr_t diff)
{
	if (runtime == NULL)
		return(0);

	void *th = dotnet_open_vm();

	json_object_add(dev->config, "diff", json_integer_new(diff));
	char *strinfo = json_serialize(dev->config);
	int32_t returnValue = 0;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceSetTime", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	if_free(strinfo);

	dotnet_close_vm(th);

	return(returnValue);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_sendUsers4J(IFDEVICE4J *dev, JSON_VALUE *users)
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	JSON_VALUE *ctx = json_array_new(1);
	JSON_VALUE *res = NULL;
	int32_t returnValue = 0;
	int r = 0;

	json_array_add(ctx, json_clone(dev->config));
	json_array_add(ctx, json_clone(users));

	char *strinfo = json_serialize(ctx);

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceSendUsers", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);

	if(ptr != NULL)
		res = json_parse_mem(ptr);

	json_value_free(ctx);
	if_free(strinfo);

	dotnet_close_vm(th);
	
	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_getUsers4J(IFDEVICE4J *dev, intptr_t none)
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	JSON_VALUE *users = NULL;
	char *strinfo = json_serialize(dev->config);
	int32_t returnValue = 0;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetUsers", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);

	if(ptr != NULL)
		users = json_parse_mem(ptr);

	if_free(strinfo);

	dotnet_close_vm(th);

	return(users);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_sendBio4J(IFDEVICE4J *dev, JSON_VALUE *users)
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	JSON_VALUE *ctx = json_array_new(1);
	JSON_VALUE *res = NULL;
	int32_t returnValue = 0;
	int r = 0;

	json_array_add(ctx, json_clone(dev->config));
	json_array_add(ctx, json_clone(users));

	char *strinfo = json_serialize(ctx);

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceSendBio", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);

	if(ptr != NULL)
		res = json_parse_mem(ptr);

	json_value_free(ctx);
	if_free(strinfo);

	dotnet_close_vm(th);
	
	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_getBio4J(IFDEVICE4J *dev, JSON_VALUE *users)
{
	if (runtime == NULL)
		return(0);

	void *th = dotnet_open_vm();

	JSON_VALUE *ctx = json_array_new(1);
	int32_t returnValue = 0;
	int r = 0;

	json_array_add(ctx, json_clone(dev->config));
	json_array_add(ctx, json_clone(users));

	char *strinfo = json_serialize(ctx);

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetBio", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);
	if_free(strinfo);

	if(ptr == NULL)
		users = NULL;
	else
		users = json_parse_mem(ptr);

	json_value_free(ctx);

	dotnet_close_vm(th);

	return(users);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * dotnet_getEvents4J(IFDEVICE4J *dev, intptr_t nsr)
{
	if (runtime == NULL)
		return(NULL);

	void *th = dotnet_open_vm();

	json_object_add(dev->config, "nsr", json_integer_new(nsr));
	char *strinfo = json_serialize(dev->config);
	int32_t returnValue = 0;
	int r = 0;

	r = callDefaultDotNetMethod(runtime, dllname, "ifponto.NativeDevice", "IfaceGetEvents", strinfo, &returnValue);
	if (r != S_OK)
		verbose(stderr, "Error Execute DefaultApp: %X\n", r);

	char *ptr = dotnet_address_recovery(strinfo, returnValue);
	if_free(strinfo);

	if (ptr == NULL)
		return(NULL);

	JSON_VALUE *offs = json_parse_mem(ptr);

	dotnet_close_vm(th);

	return(offs);
}
// ///////////////////////////////////////////////////////////////////// //
void dotnet_free4J(IFDEVICE4J *dev)
{
}
// ///////////////////////////////////////////////////////////////////// //
int dotnet_online(IFDEVICE4J *dev, IFDEVICE4J_online_callback callback, void *context)
{
	return(5);
}
// ///////////////////////////////////////////////////////////////////// //
void dotnet_init(IFDEVICE4J *dev)
{
	dev->getInfo = (IFDEVICE4J_getInfo) dotnet_getInfo4J;
	dev->getTime = (IFDEVICE4J_getTime) dotnet_getTime4J;
	dev->setTime = dotnet_setTime4J;
	dev->sendUsers = dotnet_sendUsers4J;
	dev->getUsers = dotnet_getUsers4J;
	dev->sendBio = dotnet_sendBio4J;
	dev->getBio = dotnet_getBio4J;
	dev->getEvents = dotnet_getEvents4J;
	dev->free = dotnet_free4J;
	dev->online = dotnet_online;
}
// ///////////////////////////////////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //

