#include <ifractal.h>

#include "json.tab.h"
#include "json.lex.h"


//int yyparse(void *);

// ////////////////////////////////////////////////////////////
void json_push(JSON_STACK *st, JSON_VALUE *value)
{
	st->values[st->value_pos++] = value;
	//fprintf(stdout, "push: %d\n", st->value_pos);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_current(JSON_STACK *st)
{
	return(st->values[st->value_pos - 1]);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_pop(JSON_STACK *st)
{
	JSON_VALUE *val;

	val = json_current(st);
	st->value_pos--;
	//fprintf(stdout, "pop: %d\n", st->value_pos);
	return(val);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * json_unixtime_new(time_t unixtime)
{
	JSON_VALUE *val = json_value_new(JSON_DATETIME_TYPE, &unixtime);

	return(val);
}
// ////////////////////////////////////////////////////////////
_PUBLIC time_t json_get_unixtime(JSON_VALUE *val)
{
	if (val == NULL)
		return(0);

	return(val->unixtime);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_string_new(char *str)
{
	JSON_VALUE *val = json_value_new(JSON_STRING_TYPE, str);

	return(val);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * json_integer_new(int i)
{
	JSON_VALUE *val;
	char aux[PATH_LEN];

	snprintf(aux, PATH_LEN, "%d", i);
	val = json_value_new(JSON_INTEGER_TYPE, aux);

	return(val);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * json_number_new(double d)
{
	JSON_VALUE *val;
	char aux[PATH_LEN];
	
	snprintf(aux, PATH_LEN, "%lf", d);
	val = json_value_new(JSON_NUMBER_TYPE, aux);

	return(val);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_value_new(JSON_TYPE type, void *value)
{
	JSON_VALUE *val;

	switch(type)
	{
		case JSON_NULL_TYPE:
		case JSON_INTEGER_TYPE:
		case JSON_NUMBER_TYPE:
		case JSON_STRING_TYPE:
		case JSON_BOOLEAN_TYPE:
			val = if_malloc(sizeof(JSON_VALUE));
			val->type = type;
			val->value = if_strdup((char *) value);
			break;

		case JSON_DATETIME_TYPE:	
			val = if_malloc(sizeof(JSON_VALUE));
			val->type = type;
			val->unixtime = *((time_t *) value);
			break;

		case JSON_ARRAY_TYPE:
			return(value);

		case JSON_OBJECT_TYPE:
			return(value);

		case JSON_DATA_TYPE:
			val = if_malloc(sizeof(JSON_VALUE));
			val->type = type;
			val->value = value;
			break;
	}

	return(val);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_array_new(int init_size)
{
	JSON_ARRAY *array;

	if (init_size < 1)
		init_size = 1;

	array = if_malloc(sizeof(JSON_ARRAY));

	array->type = JSON_ARRAY_TYPE;
	array->size = init_size;
	array->length = 0;

	array->values = if_malloc(sizeof(JSON_VALUE *) * init_size);

	return((JSON_VALUE *) array);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_array_add(JSON_VALUE *arr_value, JSON_VALUE *value)
{
	JSON_ARRAY *array = (JSON_ARRAY *) arr_value;

	if (arr_value->type != JSON_ARRAY_TYPE)
		return(-1);

	if (array->length >= (array->size / 2))
	{
		array->size *= 2;
		array->values = realloc(array->values, sizeof(JSON_VALUE *) * array->size);
	}

	array->values[array->length++] = value;

	return(array->length);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * json_array_remove(JSON_VALUE *arr_value, int index)
{
	JSON_ARRAY *array = (JSON_ARRAY *) arr_value;
	JSON_VALUE *aux;
	int i;

	if (arr_value->type != JSON_ARRAY_TYPE)
		return(NULL);

	if ((index < 0) || (index >= array->length))
		return(NULL);

	aux = array->values[index];
	array->length--;

	for (i = index ; i < array->length ; i++)
		array->values[i] = array->values[i + 1];

	return(aux);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_object_new(int init_size)
{
	JSON_OBJECT *object;

	if (init_size < 1)
		init_size = 1;

	object = if_malloc(sizeof(JSON_OBJECT));

	object->type = JSON_OBJECT_TYPE;
	object->size = init_size;
	object->length = 0;

	object->properties = if_malloc(sizeof(JSON_PROPERTY *) * init_size);

	return((JSON_VALUE *) object);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_object_add(JSON_VALUE *val_obj, char *key, JSON_VALUE *value)
{
	JSON_OBJECT *obj = (JSON_OBJECT *) val_obj;
	JSON_PROPERTY *prop;
	int i;

	if ((val_obj == NULL) || (val_obj->type != JSON_OBJECT_TYPE))
		return(-1);

	// Substitui valor caso "key" ja exista
	for (i = 0 ; i < obj->length ; i++)
	{
		prop = obj->properties[i];
		if (strcmp(prop->key, key) == 0)
		{
			json_value_free(prop->value);
			prop->value = value;
			return(obj->length);
		}
	}

	if (obj->length >= (obj->size / 2))
	{
		obj->size *= 2;
		obj->properties = realloc(obj->properties, sizeof(JSON_PROPERTY *) * obj->size);
	}

	prop = if_malloc(sizeof(JSON_PROPERTY));
	prop->key = if_strdup(key);
	prop->value = value;
	obj->properties[obj->length++] = prop;

	return(obj->length);
}
// ////////////////////////////////////////////////////////////
_PUBLIC int json_object_add_list(JSON_VALUE *object, char *list[])
{
	int i;

	if ((object == NULL) || (list == NULL))
		return(-1);

	for (i = 0 ; list[i] != NULL ; i += 2)
		json_object_add(object, list[i], json_string_new(list[i + 1]));

	return(i / 2);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC char * json_object_get_string(JSON_VALUE *obj, char *key)
{
	JSON_VALUE *aux;

	if ((obj == NULL) || (json_get_type(obj) != JSON_OBJECT_TYPE) || (key == NULL))
		return("");

	if ((aux = json_object_find(obj, key)) == NULL)
		return("");

	return(json_get_string(aux));
}
// ////////////////////////////////////////////////////////////
_PUBLIC int json_object_get_int(JSON_VALUE *obj, char *key)
{
	return(atoi(json_object_get_string(obj, key)));
}
// ////////////////////////////////////////////////////////////
_PUBLIC time_t json_object_get_unixtime(JSON_VALUE *obj, char *key)
{
	JSON_VALUE *aux;

	if ((obj == NULL) || (json_get_type(obj) != JSON_OBJECT_TYPE) || (key == NULL))
		return(0);

	if ((aux = json_object_find(obj, key)) == NULL)
		return(0);

	return(json_get_unixtime(aux));
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
void json_value_free(JSON_VALUE *val)
{
	if (val == NULL)
		return;

	if (val->value != NULL)
	{
		switch (val->type)
		{
			case JSON_NULL_TYPE:
			case JSON_INTEGER_TYPE:
			case JSON_NUMBER_TYPE:
			case JSON_BOOLEAN_TYPE:
			case JSON_STRING_TYPE:
				if_free(val->value);
				break;

			case JSON_ARRAY_TYPE:
				json_array_free((JSON_ARRAY *) val);
				break;

			case JSON_OBJECT_TYPE:
				json_object_free((JSON_OBJECT *) val);
				break;

			case JSON_DATETIME_TYPE:
				break;

			case JSON_DATA_TYPE:
				// nao libera (free) dados externos a estrutura.
				break;
		}
	}

	if_free(val);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * json_clone(JSON_VALUE *json)
{
	JSON_VALUE *clone;
	char *aux;

	if (json == NULL)
		return(NULL);

	aux = json_serialize(json);
	clone = json_parse_mem(aux);
	if_free(aux);

	return(clone);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void json_array_free(JSON_ARRAY *array)
{
	int i;

	if (array == NULL)
		return;

	if (array->values != NULL)
	{
		for (i = 0 ; i < array->length ; i++)
			json_value_free(array->values[i]);

		if_free(array->values);
	}
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void json_object_free(JSON_OBJECT *obj)
{
	int i;

	if (obj == NULL)
		return;

	if (obj->properties != NULL)
	{
		for (i = 0 ; i < obj->length ; i++)
		{
			json_value_free(obj->properties[i]->value);
			if_free(obj->properties[i]->key);
			if_free(obj->properties[i]);
		}
		if_free(obj->properties);
	}
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_object_remove(JSON_VALUE *val, char *key)
{
	JSON_OBJECT *obj;
	int i, qty = 0;

	if (val == NULL)
		return(-1);

	if (val->type != JSON_OBJECT_TYPE)
		return(-2);

	obj = (JSON_OBJECT *) val;
	if (obj == NULL)
		return(-3);

	if (obj->properties == NULL)
		return(0);

	for (i = 0 ; i < obj->length ; i++)
	{
		if (strcmp(obj->properties[i]->key, key) == 0)
		{
			json_value_free(obj->properties[i]->value);
			if_free(obj->properties[i]->key);
			if_free(obj->properties[i]);
			obj->length--;
			qty++;
			break;
		}
	}

	for ( ; i < obj->length ; i++)
		obj->properties[i] = obj->properties[i + 1];

	return(qty);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_object_iter(JSON_VALUE *val, JSON_OBJECT_ITER_CALLACK callback, void *user_data)
{
	JSON_PROPERTY *prop;
	JSON_OBJECT *obj;
	int i;

	if ((val == NULL) || (callback == NULL))
		return(-1);

	if (val->type != JSON_OBJECT_TYPE)
		return(-2);

	obj = (JSON_OBJECT *) val;
	if (obj->properties == NULL)
		return(0);

	for (i = 0 ; i < obj->length ; i++)
	{
		prop = obj->properties[i];

		if (callback(prop->key, prop->value, user_data))
			break;
	}

	return(i);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_array_iter(JSON_VALUE *val, JSON_ARRAY_ITER_CALLACK callback, void *user_data)
{
	JSON_ARRAY *array;
	int i;

	if ((val == NULL) || (callback == NULL))
		return(-1);

	if (val->type != JSON_ARRAY_TYPE)
		return(-2);

	array = (JSON_ARRAY *) val;
	for (i = 0 ; i < array->length ; i++)
	{
		if (callback(array->values[i], user_data))
			break;
	}

	return(i);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_TYPE json_get_type(JSON_VALUE *val)
{
	if (val == NULL)
		return(JSON_NULL_TYPE);

	return(val->type);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_get_int(JSON_VALUE *val)
{
	if (val == NULL)
		return(0);

	if (
		(val->type != JSON_INTEGER_TYPE) &&
		(val->type != JSON_NUMBER_TYPE) &&
		(val->type != JSON_STRING_TYPE)
		)
		return(0);

	return(atoi(val->value));
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
double json_get_double(JSON_VALUE *val)
{
	if (val == NULL)
		return(0.0);

	if (val->type != JSON_NUMBER_TYPE)
		return(0.0);

	return(atof(val->value));
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_get_boolean(JSON_VALUE *val)
{
	if (val == NULL)
		return(0);

	if (val->type != JSON_BOOLEAN_TYPE)
		return(0);

	return(atoi(val->value));
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * json_get_string(JSON_VALUE *val)
{
	if (val == NULL)
		return("");

	switch(val->type)
	{
		case JSON_INTEGER_TYPE: 
		case JSON_NUMBER_TYPE: 
		case JSON_BOOLEAN_TYPE:
		case JSON_STRING_TYPE: 
			return(val->value);

		default:
			break;
	}

	return("");
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_object_find(JSON_VALUE *val, char *key)
{
	JSON_VALUE *element = NULL;
	JSON_PROPERTY *prop;
	JSON_OBJECT *obj;
	int i;

	if ((val == NULL) || (key == NULL))
		return(NULL);

	if (json_get_type(val) != JSON_OBJECT_TYPE)
		return(NULL);

	obj = (JSON_OBJECT *) val;
	for (i = 0 ; i < obj->length ; i++)
	{
		prop = obj->properties[i];
		if (strcmp(prop->key, key) == 0)
		{
			element = prop->value;
			break;
		}
	}

	return(element);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * json_array_index(JSON_VALUE *val, int index)
{
	JSON_ARRAY *array;

	if (val == NULL)
		return(NULL);

	if (json_get_type(val) != JSON_ARRAY_TYPE)
		return(NULL);

	array = (JSON_ARRAY *) val;
	if ((index < 0) || (index >= array->length))
		return(NULL);

	return(array->values[index]);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int json_array_length(JSON_VALUE *value)
{
	return(((JSON_ARRAY *) value)->length);
}
// ////////////////////////////////////////////////////////////
int json_object_length(JSON_VALUE *value)
{
	return(((JSON_OBJECT *) value)->length);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
void adjust_string(char *str)
{
	int i;
	for (i = 0 ; str[i] != 0 ; i++)
		if (str[i] == '"')
			str[i] = ' ';
}
// ////////////////////////////////////////////////////////////
char * json_serialize_realloc(
	_INOUT char **buf, 
	_IN JSON_VALUE *val, 
	_INOUT int *size, 
	_IN char *current, 
	_IN char *content)
{
	int pos, len;

	len = strlen(content);
	pos = len + (current - *buf);

	if (pos < (*size / 2))
		return(current);

	*size *= 2;
	*buf = realloc(*buf, *size);
	current = *buf + pos;

	current -= len;
	return(current);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE int json_serialize_scape(_INOUT STRING_BUFFER *sb, char *value)
{
	char *p, buf[] = {0,0};
	char tmp[10];
	int n;

	for (n = 0, p = value ; *p != 0 ; p++)
	{
		if ((*p == '\"') || (*p == '\\'))
		{
			string_append(sb, "\\");
			n++;
		}

		if ((*p == 0xC2) || (*p == 0xC3))
		{
			p++;
			snprintf(tmp, sizeof(tmp), "\\u00%02x", *p);
			string_append(sb, tmp);
		}
		else
		{
			buf[0] = *p;
			string_append(sb, buf);
		}
	}

	return(n);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE void json_serialize_in(_IN JSON_VALUE *value, _INOUT STRING_BUFFER *sb)
{
	char date_format[] = "\"%04d-%02d-%02d %02d:%02d:%02d\"";
	char date_buf[sizeof(date_format)];
	JSON_ARRAY *array;
	JSON_OBJECT *obj;
	char *buf = if_malloc(PATH_LEN);
	struct tm *dt;
	int i;

	switch (value->type)
	{
		case JSON_DATA_TYPE:
			snprintf(buf, PATH_LEN, "%ld", (intptr_t) value->value);
			string_append(sb, buf);
			break;

		case JSON_NULL_TYPE:
			string_append(sb, "null");
			break;

		case JSON_INTEGER_TYPE: 
		case JSON_NUMBER_TYPE: 
		case JSON_BOOLEAN_TYPE:
			string_append(sb, value->value);
			break;

		case JSON_STRING_TYPE: 
			string_append(sb, "\"");
			json_serialize_scape(sb, value->value);
			string_append(sb, "\"");
			break;

		case JSON_DATETIME_TYPE:
			dt = localtime(&(value->unixtime));
			snprintf(date_buf, sizeof(date_buf), date_format,
				dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday, 
				dt->tm_hour, dt->tm_min, dt->tm_sec);
			string_append(sb, date_buf);
			break;

		case JSON_ARRAY_TYPE: 
			array = (JSON_ARRAY *) value;
			string_append(sb, "[");
			for (i = 0 ; i < array->length ; i++)
			{
				if (i > 0)
					string_append(sb, ",");

				json_serialize_in(array->values[i], sb);
			}
			string_append(sb, "]");
			break;

		case JSON_OBJECT_TYPE:
			obj = (JSON_OBJECT *) value;
			string_append(sb, "{");
			for (i = 0 ; i < obj->length ; i++)
			{
				if (i > 0)
					string_append(sb, ",");

				// TODO - permitir que a chave possua o charater aspas
				adjust_string(obj->properties[i]->key);

				string_append(sb, "\"");
				string_append(sb, obj->properties[i]->key);
				string_append(sb, "\":");

				json_serialize_in(obj->properties[i]->value, sb);
			}
			string_append(sb, "}");
			break;
	}

	if_free(buf);
}
// ///////////////////////////////////////////////////////// //
char * json_serialize(JSON_VALUE *value)
{
	STRING_BUFFER *sb;
	char *buf;

	if (value == NULL)
		return(if_strdup("null"));

	sb = string_new("");

	json_serialize_in(value, sb);

	buf = if_strdup(string_get_text(sb));

	string_free(sb);

	return(buf);
}
// ////////////////////////////////////////////////////////////
_PUBLIC int json_serialize_file(JSON_VALUE *value, char *filename)
{
	STRING_BUFFER *sb;
	FILE *fd;
	int n;

	fd = fopen(filename, "w");
	if (fd == NULL)
		return(-1);

	if (value == NULL)
	{
		n = fprintf(fd, "null");
		fclose(fd);
		return(n);
	}

	sb = string_new("");
	json_serialize_in(value, sb);
	n = fprintf(fd, "%s", string_get_text(sb));
	string_free(sb);
	fclose(fd);

	return(n);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_object_new_list(_IN char *list[])
{
	JSON_VALUE *obj, *aux;
	int i;

	if (list == NULL)
		return(NULL);

	obj = json_object_new(2);
	for (i = 0 ; list[i] != NULL ; i += 2)
	{
		if (list[i + 1] == NULL)
			break;
		
		aux = json_string_new(list[i + 1]);
		json_object_add(obj, list[i], aux);
	}

	return(obj);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
void json_isert_tabs(int qty)
{
	int i;

	for (i = 0 ; i < qty ; i++)
		fprintf(stdout, "\t");
}
// ////////////////////////////////////////////////////////////
char * json_type2txt(JSON_TYPE type)
{
	switch(type)
	{
		case JSON_INTEGER_TYPE: return("INTEGER");
		case JSON_NUMBER_TYPE: return("NUMBER");
		case JSON_BOOLEAN_TYPE: return("BOOLEAN");
		case JSON_STRING_TYPE: return("STRING");
		case JSON_DATETIME_TYPE: return("DATETIME");
		case JSON_ARRAY_TYPE: return("ARRAY");
		case JSON_OBJECT_TYPE: return("OBJECT");
		case JSON_DATA_TYPE: return("POINTER");
		case JSON_NULL_TYPE: return("NULL");
	}

	return("### tipo invalido ###");
}
// ////////////////////////////////////////////////////////////
void json_show_value_in(JSON_VALUE *value, int deep)
{
	JSON_ARRAY *array;
	JSON_OBJECT *obj;

	int i;

	if (value == NULL)
		return;

	json_isert_tabs(deep);
		
	fprintf(stdout, "%s: ", json_type2txt(value->type));
	switch (value->type)
	{
		case JSON_NULL_TYPE:
			fprintf(stdout, "NULL\n");
			break;

		case JSON_DATA_TYPE:
			fprintf(stdout, "%lX\n", (intptr_t) value->value);
			break;

		case JSON_INTEGER_TYPE: 
		case JSON_NUMBER_TYPE: 
		case JSON_BOOLEAN_TYPE:
		case JSON_STRING_TYPE: 
			fprintf(stdout, "%s\n", value->value);
			break;

		case JSON_DATETIME_TYPE:
			fprintf(stdout, "%ld\n", value->unixtime);
			break;

		case JSON_ARRAY_TYPE: 
			array = (JSON_ARRAY *) value;
			fprintf(stdout, "\n");
			for (i = 0 ; i < array->length ; i++)
				json_show_value_in(array->values[i], deep + 1);
			break;

		case JSON_OBJECT_TYPE:
			obj = (JSON_OBJECT *) value;
			fprintf(stdout, "\n");
			for (i = 0 ; i < obj->length ; i++)
			{
				json_isert_tabs(deep + 1);
				fprintf(stdout, "KEY: %s\n", obj->properties[i]->key);
				json_show_value_in(obj->properties[i]->value, deep + 2);
			}
			break;
	}
}
// ////////////////////////////////////////////////////////////
void json_show_value(JSON_VALUE *value)
{
	json_show_value_in(value, 0);
}
// ////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////
void lex_init(JSON_STACK *stack, yyscan_t *scanner)
{
	yylex_init(scanner);
	yyset_extra(stack, *scanner);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
void json_input(char *buf, int *result, int max_size, void *extra)
{
	JSON_STACK *stack = extra;
	int c;

	c = stack->context.next_char(&(stack->context));
	if (c == EOF)
	{
		*result = 0;
		return;
	}

	*buf = c;
	*result = 1;
}
// ////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////
int json_fd_next_char(JSON_LEX_CONTEXT *context)
{
	char byte[2];

	if (fread(byte, 1, 1, context->fd_in) < 1)
		return(EOF);

	context->mem_pos_in++;

	return(byte[0]);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_parse_file(FILE *fd)
{
	yyscan_t scanner;
	JSON_STACK stack;
	memset(&stack, 0, sizeof(JSON_STACK));

	if (fd == NULL)
		return(NULL);

	lex_init(&stack, &scanner);

	stack.context.next_char = json_fd_next_char;
	stack.context.fd_in = fd;
	stack.scanner = scanner;

	if (yyparse(&stack))
	{
		json_value_free(stack.val);
		return(NULL);
	}

	yylex_destroy(scanner);

	return(stack.val);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC int json_make_list_param(_IN JSON_VALUE *obj, _INOUT char **list)
{
	JSON_VALUE *aux;
	int i;

	if ((obj == NULL) || (list == NULL))
		return(-1);

	if (json_get_type(obj) != JSON_OBJECT_TYPE)
		return(-2);

	for (i = 0 ; list[i] != NULL ; i += 2)
        {
		aux = json_object_find(obj, list[i]);
		if (aux != NULL)
			list[i + 1] = json_get_string(aux);
        }

	return(i / 2);
}
// ////////////////////////////////////////////////////////////
_PUBLIC char * json_get_list_param(_IN char **list, _IN char *key)
{
	int i;

	if ((list == NULL) || (key == NULL))
		return(NULL);

	for (i = 0 ; list[i] != NULL ; i += 2)
	{
		if (strcmp(list[i], key) == 0)
			return(list[i + 1]);
	}

	return("");
}
// ////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////
int json_mem_next_char(JSON_LEX_CONTEXT *context)
{
	int byte = (int) context->mem_in[context->mem_pos_in++];

	//while ((byte == '\\') && (context->mem_in[context->mem_pos_in] == '"'))
	//{
	//	context->mem_pos_in++;
	//	byte = (int) context->mem_in[context->mem_pos_in++];
	//}

	if (byte == 0)
		return(EOF);

	return(byte);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * json_parse_mem(char *buf)
{
	yyscan_t scanner;
	JSON_STACK stack;
	memset(&stack, 0, sizeof(JSON_STACK));

	lex_init(&stack, &scanner);

	stack.context.next_char = json_mem_next_char;
	stack.context.mem_in = buf;
	stack.context.mem_pos_in = 0;
	stack.scanner = scanner;

	if (yyparse(&stack))
		return(NULL);

	yylex_destroy(scanner);

	return(stack.val);
}
// ////////////////////////////////////////////////////////////



#ifdef STANDALONE

#define DESCRIPT	"Biblioteca JSON."


// ////////////////////////////////////////////////////////////
void test()
{
	JSON_VALUE *val, *array, *obj;
	char buf[20], *p;
	int i, j;

	obj = json_object_new(2);
	for(j = 0 ; j < 200 ; j++)
	{
		array = json_array_new(10);
		for (i = 0 ; i < 1000 ; i++)
		{
			sprintf(buf, "%d", i);
			val = json_value_new(JSON_INTEGER_TYPE, buf);
			json_array_add(array, val);
		}

		sprintf(buf, "\"\"obj_%d", j);
		json_object_add(obj, buf, array);
	}

	p = json_serialize(obj);
	json_value_free(obj);

	printf("%s\n", p);
	if_free(p);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int validate(char *file)
{
	JSON_VALUE *json;
	FILE *fd;

	fd = fopen(file, "r");
	if (fd == NULL)
	{
		fprintf(stderr, "Falha ao tentar abrir: '%s'.\n", file);
		return(-1);
	}

	json = json_parse_file(fd);
	fclose(fd);

	if (json != NULL)
	{
		json_show_value(json);
		json_value_free(json);
        
		fprintf(stdout, "OK\n");
	}

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int object_callback(char *key, JSON_VALUE *val, void *user_data)
{
	fprintf(stdout, "%s: %s\n", key, json_type2txt(json_get_type(val)));
	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int array_callback(JSON_VALUE *val, void *user_data)
{
	fprintf(stdout, "%s\n", json_type2txt(json_get_type(val)));
	return(0);
}
// ////////////////////////////////////////////////////////////



static IF_GETOPT opts[] = {
	{0, 'i', IF_GETOPT_TYPE_STRING, "test-stdin", "", 0, "Valida JSON por stdin."},
	{0, 'v', IF_GETOPT_TYPE_STRING, "validate", "", 0, "Valida arquivo json."},
	{0, 'o', IF_GETOPT_TYPE_STRING, "object-iter", "", 0, "Testa iterador de objetos."},
	{0, 'a', IF_GETOPT_TYPE_STRING, "array-iter", "", 0, "Testa iterador de vetor."},
	{0, 't', IF_GETOPT_TYPE_NONE, "test", "", 0, "Testes de carga."},
	{0, 0, 0, 0, 0, 0, 0}
};

// ////////////////////////////////////////////////////////////
int main (int argc, char **argv)
{
	JSON_VALUE *json;
	char *param;
	int r;

	r = if_getopt(opts, argc, argv);
	if (	(r != 0) || 
		((!if_getopt_isChecked(opts, "validate")) &&
		(!if_getopt_isChecked(opts, "object-iter")) && 
		(!if_getopt_isChecked(opts, "array-iter")) && 
		(!if_getopt_isChecked(opts, "test-stdin")) && 
		(!if_getopt_isChecked(opts, "test")))
		)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(opts);

		fprintf(stderr, "\nUso:\n\tshell$ %s -v xyz.json\n\n", argv[0]);
		return (1);
	}

	if (if_getopt_isChecked(opts, "validate"))
	{
		param = if_getopt_getValue(opts, "validate");
		return(validate(param));
	}

	if (if_getopt_isChecked(opts, "object-iter"))
	{
		param = if_getopt_getValue(opts, "object-iter");

		json = json_parse_mem(param);
		json_object_iter(json, object_callback, NULL);

		printf("\n\n");
		json_object_remove(json, "key");
		json_object_iter(json, object_callback, NULL);

		json_value_free(json);
	}

	if (if_getopt_isChecked(opts, "array-iter"))
	{
		param = if_getopt_getValue(opts, "array-iter");

		json = json_parse_mem(param);
		json_array_iter(json, array_callback, NULL);
		json_value_free(json);
	}

	if (if_getopt_isChecked(opts, "test-stdin"))
	{
		json = json_parse_file(stdin);
		json_show_value(json);
		json_value_free(json);
	}

	if (if_getopt_isChecked(opts, "test"))
		test();

	return (0);
}
// ////////////////////////////////////////////////////////////

#endif


