#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ifractal.h>

#include <core.h>
#include <cgi.h>


#define DESCRIPT	"Processador de arquivos de posicao fixa."


// ////////////////////////////////////////////////////////////
_PRIVATE int get_offset(char *line, int position)
{
	int i, off = 0;
	unsigned char c;

	for (i = 0 ; (line[i] != 0) && (i < (position + off)) ; i++)
	{
		c = line[i];
		if ((c == 0xC2) || (c == 0xC3))
			off++;
	}

	return(off + position);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_compile_json_funcs(IF_PARSER_FUNC_DEF *func_defs)
{
	JSON_VALUE *config, *compiled = NULL;
	int i;

	compiled = json_array_new(2);

	if (func_defs == NULL)
		return(compiled);

	for (i = 0 ; func_defs[i].json_config != NULL ; i++)
	{
		if (func_defs[i].func == NULL)
			continue;

		config = json_parse_mem(func_defs[i].json_config);

		if (config == NULL)
		{
			fprintf(stderr, "compile json func invalida: '%s'.\n", func_defs[i].json_config);
			continue;
		}

		if (json_get_type(config) != JSON_OBJECT_TYPE)
		{
			fprintf(stderr, "json func invalida deve ser um objeto: '%s'.\n", 
				func_defs[i].json_config);
			continue;
		}

		json_object_add(config, "func", json_value_new(JSON_DATA_TYPE, func_defs[i].func));

		json_array_add(compiled, config);
	}

	return(compiled);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
IF_PARSER_MODULE * parser_module_new(char *name)
{
	IF_PARSER_MODULE *module;

	module = if_malloc(sizeof(IF_PARSER_MODULE));

	strncpy(module->name, name, PATH_LEN);
	module->init = NULL;
	module->finalize = NULL;
	module->funcs = NULL;
	module->next = NULL;

	return(module);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
char * show_json_funcs(IF_PARSER_MODULE *mods)
{
	JSON_VALUE *mod_array, *obj;
	IF_PARSER_MODULE *mod;
	char *json_str;

	mod_array = json_array_new(2);
	for (mod = mods ; mod != NULL ; mod = mod->next)
	{
		obj = json_object_new(2);
		json_object_add(obj, "name", json_string_new(mod->name));
		json_object_add(obj, "funcs", mod->funcs);

		json_array_add(mod_array, obj);
	}

	json_str = json_serialize(mod_array);

	json_value_free(mod_array);

	return(json_str);
}
// ////////////////////////////////////////////////////////////



IF_PARSER_MODULE *modules = NULL;

// ////////////////////////////////////////////////////////////
int parser_init_modules()
{
	IF_PARSER_MODULE *mod;

	mod = parser_module_new("core");
	mod->init = parser_module_func_init;
	mod->finalize = parser_module_func_finalize;

	mod->funcs = mod->init();

	modules = mod;

	return(1);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int parser_finalize_modules()
{
	json_value_free(modules->funcs);
	modules->finalize();

	if_free(modules);

	return(0);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
int if_get_content(_IN FILE *fd, _OUT char **ref)
{
	int size = BUFFER_LEN;
	int len, n;
	unsigned char *buf = if_malloc(size);
	unsigned char *p;

	for (p = buf, len = p - buf ; (n = fread(p, 1, size - len, fd)) > 0 ; )
	{
		p += n;
		len = p - buf;
		if (len > (size / 2))
		{
			size *= 2;
			buf = realloc(buf, size);
			p = buf + len;
		}
	}

	*ref = (char *) buf;
	return(len);
}
// ////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////
void addGlobals(_INOUT JSON_VALUE *result, _IN JSON_VALUE *data_source, _IN JSON_VALUE *value)
{
	JSON_VALUE *globals, *ds_name, *aux;
	char *name;

	globals = json_object_find(result, "globals");
	ds_name = json_object_find(data_source, "name");
	name = json_get_string(ds_name);

	aux = json_object_find(globals, name);
	if (aux != NULL)
		json_object_remove(globals, name);

	json_object_add(globals, name, value);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int checkLineMultiKey(char *line, char *multikey)
{
	int id_len, tk_len, i;
	char *tk[20];
	char *id;
	int ret = 0;

	id = if_strdup(multikey);
	tk_len = tokenizer('|', id, tk, 19);
	for (ret = 0, i = 0 ; i < tk_len ; i++)
	{
		id_len = strlen(tk[i]);
		if (strncmp(line, tk[i], id_len) == 0)
		{
			ret = 1;
			break;
		}
	}

	if_free(id);

	return(ret);
}
// ////////////////////////////////////////////////////////////
int checkLineKeyIter(JSON_VALUE *value, void *user_data)
{
	JSON_VALUE *aux;
	char *line = (char *) user_data;
	int position;
	char *key;

	aux = json_object_find(value, "initial_char");
	position = json_get_int(aux) - 1;
	if (position < 0)
		return(1);

	position = get_offset(line, position);
	aux = json_object_find(value, "match");
	key = json_get_string(aux);
	if (key == NULL)
		return(2);

	if ((position + strlen(key)) > strlen(line))
		return(3);

	if (!checkLineMultiKey(line + position, key))
		return(4);

	return(0);
}
// ////////////////////////////////////////////////////////////
int checkLineKey(_IN JSON_VALUE *obj, _IN char *line)
{
	JSON_VALUE *keys;
	int k;

	keys = json_object_find(obj, "positional_line_keys");
	if (keys == NULL)
		return(0);

	k = json_array_iter(keys, checkLineKeyIter, line);
	if (k != json_array_length(keys))
		return(0);

	return(1);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
IF_PARSER_FUNC_POINTER parser_getFunc(int id)
{
	IF_PARSER_FUNC_POINTER func_pointer = NULL;
	IF_PARSER_MODULE *mod = NULL;
	JSON_VALUE *func;
	int i, fid;

	for (mod = modules ; mod != NULL ; mod = mod->next)
	{
		for (i = 0 ; i < json_array_length(mod->funcs) ; i++)
		{
			func = json_array_index(mod->funcs, i);
			fid = json_get_int(json_object_find(func, "id"));

			if (id != fid)
				continue;

			func_pointer = (IF_PARSER_FUNC_POINTER) json_object_find(func, "func")->value;
		} 
	}
	return(func_pointer);
}
// ////////////////////////////////////////////////////////////
void parser_apply_functions(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token)
{
	JSON_VALUE *funcs, *func, *params, *func_id;
	IF_PARSER_FUNC_POINTER func_pointer;
	int i;

	funcs = json_object_find(token, "formatting_function_calls");
	if (funcs == NULL)
		return;

	for (i = 0 ; i < json_array_length(funcs) ; i++)
	{
		func = json_array_index(funcs, i);
		func_id = json_object_find(func, "function_definition_id");
		if (func_id == NULL)
			continue;

		func_pointer = parser_getFunc(json_get_int(func_id));
		if (func_pointer == NULL)
			continue;

		params = json_object_find(func, "function_call_parameters");
		func_pointer(data, result, record, templ, data_source, token, params);
	}
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void addException(_INOUT JSON_VALUE *result, _IN char *description, _INOUT JSON_VALUE *data_source, int line)
{
	JSON_VALUE *obj = json_object_new(3);
	JSON_VALUE *exceptions, *aux;
	char *name = json_object_get_string(data_source, "name");

	json_object_add(obj, name, json_string_new(description));

	aux = json_integer_new(line);
	json_object_add(obj, "line", aux);

	exceptions = json_object_find(result, "exceptions");
	json_array_add(exceptions, obj);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void get_initial_and_length(_IN JSON_VALUE *token, _INOUT int *position, _INOUT int *len)
{
	JSON_VALUE *aux;

	*position = 0;
	*len = 0;
	aux = json_object_find(token, "initial_char");
	if (aux != NULL)
		*position = json_get_int(aux) - 1;

	if (*position < 0)
	{
		*len = 0;
		return;
	}

	aux = json_object_find(token, "content_length");
	if (aux != NULL)
		*len = json_get_int(aux);

	if (*len < 1)
	{
		*len = 0;
		return;
	}
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * get_line_value(
	_IN char *line,
	_INOUT JSON_VALUE *result, 
	_IN JSON_VALUE *data_source, 
	_IN int line_n, 
	_IN int position, _IN int len)
{
	char msg[PATH_LEN], *data;
	int i, line_len;

	line_len = strlen(line);
	for (i = line_len - 1 ; (line[i] == '\r') || (line[i] == '\n') ; i--)
		line_len--;

	if (position > line_len)
	{
		snprintf(msg, PATH_LEN, "Parametros inconsistentes: inicio (%d) eh maior que o tamanho linha (%d).", 
			position + 1, line_len);

		addException(result, msg, data_source, line_n);
		return(NULL);
	}

	if ((position + len) > line_len)
	{
		snprintf(msg, PATH_LEN, "Campo truncado: inicio (%d) + tamanho (%d) eh maior que a linha (%d).", 
			position + 1, len, line_len);

		addException(result, msg, data_source, line_n);
		len = line_len - position;
	}

	if (len < 1)
		return(NULL);

	data = if_malloc(len + BUFFER_LEN);
	for (i = 0 ; i < len ; i++)
		data[i] = line[i + position];
	data[i] = 0;

	return(data);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void processFixed(
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token, 
	_IN char *line,
	_IN int line_n)
{
	JSON_VALUE *aux, *value;

	aux = json_object_find(token, "value");
	value = json_string_new(json_get_string(aux));

	addGlobals(result, data_source, value);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PRIVATE int processLocal_insert_multi(_INOUT JSON_VALUE *record, _IN char *field_name, _IN JSON_VALUE *content)
{
	char buf[PATH_LEN];
	JSON_VALUE *aux;
	int c = 0;

	snprintf(buf, PATH_LEN, "%s", field_name);
	aux = json_object_find(record, buf);

	if (aux != NULL)
	{
		snprintf(buf, PATH_LEN, "%s%03d", field_name, 1);
		aux = json_object_find(record, buf);
		if (aux == NULL)
		{
			aux = json_clone(json_object_find(record, field_name));
			json_object_add(record, buf, aux);
		}

		for (c = 2 ; aux != NULL ; c++)
		{
			snprintf(buf, PATH_LEN, "%s%03d", field_name, c);
			aux = json_object_find(record, buf);
		}
	}

	json_object_add(record, buf, content);

	if (c > 1)
	{
		// Acrescenta o ultimo
		aux = json_clone(content);
		snprintf(buf, PATH_LEN, "%s%03d", field_name, 999);
		json_object_add(record, buf, aux);
	}

	return(c);
}
// ////////////////////////////////////////////////////////////
_PRIVATE void processLocal(
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token, 
	_IN char *line,
	_IN int line_n)
{
	int position, len, off, bytepos;
	JSON_VALUE *aux, *name;
	char *data;

	if (!checkLineKey(token, line))
		return;

	get_initial_and_length(token, &position, &len);
	if (len < 1)
	{
		//addException(result, "parametros incorretos: inicio, tamanho.", data_source, line_n);
		//fprintf(stderr, "parametros incorretos: inicio, tamanho.\n");
		return;
	}

	if (record == NULL)
	{
		//addException(result, "registro nao inicializado", data_source, line_n);
		//fprintf(stderr, "registro nao inicializado\n");
		return;
	}

	bytepos = get_offset(line, position);
	off = get_offset(line, position + len);
	data = get_line_value(line, result, data_source, line_n, bytepos, off - bytepos);
	if (data == NULL)
		return;

	parser_apply_functions(data, result, record, templ, data_source, token);

	aux = json_string_new(data);
	if_free(data);

	name = json_object_find(data_source, "name");
	processLocal_insert_multi(record, json_get_string(name), aux);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void processGlobal(
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token, 
	_IN char *line,
	_IN int line_n)
{
	int position, len, bytepos, off;
	JSON_VALUE *value;
	char *data;

	if (!checkLineKey(token, line))
		return;

	get_initial_and_length(token, &position, &len);
	if (len < 1)
	{
		addException(result, "parametros incorretos: inicio, tamanho.", data_source, line_n);
		return;
	}

	bytepos = get_offset(line, position);
	off = get_offset(line, position + len);
	data = get_line_value(line, result, data_source, line_n, bytepos, off - bytepos);
	//data = get_line_value(line, result, data_source, line_n, position, len);
	if (data == NULL)
		return;

	parser_apply_functions(data, result, record, templ, data_source, token);

	value = json_string_new(data);
	if_free(data);

	addGlobals(result, data_source, value);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void processGenerated(
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token, 
	_IN char *line,
	_IN int line_n)
{
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void processReference(
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token, 
	_IN char *line,
	_IN int line_n)
{
}
// ////////////////////////////////////////////////////////////



static PARSER_TYPE_MAP token_types[] = {
	{"FixedDataToken", processFixed},
	{"LocalDataToken", processLocal},
	{"GlobalDataToken", processGlobal},
	{"GeneratedDataToken", processGenerated},
	{"ReferenceDataToken", processReference},
	{NULL, NULL}
};

// ////////////////////////////////////////////////////////////
int data_sourceIter(JSON_VALUE *data_source, void *user_data)
{
	JSON_VALUE *tokens, *token, *type;
	void **params = user_data;
	JSON_VALUE *templ = (JSON_VALUE *) params[0];
	JSON_VALUE *result = (JSON_VALUE *) params[1];
	JSON_VALUE *record = (JSON_VALUE *) params[2];
	char *line = (char *) params[3];
	int line_n = (int) params[4];
	int i, len;

	tokens = json_object_find(data_source, "data_tokens");
	// TODO - Tratar multiplos tokens
	token = json_array_index(tokens, 0);
	type = json_object_find(token, "type");

	for (i = 0 ; token_types[i].type != NULL ; i++)
	{
		len = strlen(token_types[i].type);
		if (strncmp(json_get_string(type), token_types[i].type, len) == 0)
		{
			token_types[i].process(result, record, templ, data_source, token, line, line_n);
		}
	}

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * processLine(
	_IN JSON_VALUE *templ, 
	_INOUT JSON_VALUE *result, 
	_INOUT JSON_VALUE *record, 
	_IN char *line,
	_IN int line_n)
{
	void *params[5];

	JSON_VALUE *data_sources = json_object_find(templ, "data_sources");

	params[0] = templ;
	params[1] = result;
	params[2] = record;
	params[3] = line;
	params[4] = (void *) line_n;
	json_array_iter(data_sources, data_sourceIter, params);

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * runGetLine(_INOUT char **ref)
{
	static int sum = 0;
	char *p, *dupline;
	int len;
	
	for (p = *ref ; (*p != 0) && (*p != '\n') ; p++)
		;

	if (*p == 0)
		return(NULL);

	len = 1 + (p - *ref);
	dupline = if_malloc(len + 1);
	memcpy(dupline, *ref, len);
	dupline[len] = 0;

	sum +=  len;
	*ref = p + 1;
	return(dupline);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_init_result_file(_IN char *txt_content, _IN int content_length)
{
	JSON_VALUE *obj, *aux;
	int i, lines;

	for (i = 0, lines = 0 ; i < content_length ; i++)
		if (txt_content[i] == '\n')
			lines++;

	obj = json_object_new(2);

	aux = json_integer_new(i);
	json_object_add(obj, "length", aux);

	aux = json_integer_new(lines);
	json_object_add(obj, "lines_qty", aux);

	return(obj);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_init_result(_IN char *txt_content, _IN int content_length)
{
	int bom_offset = 0;
	BOM_CODE *bom;

	JSON_VALUE *aux, *result = NULL;

	result = json_parse_mem("{'globals':{'_':0}, 'exceptions':[], 'records':[]}");

	bom = if_get_BOM(txt_content);
	if (bom != NULL)
	{
		fprintf(stderr, "BOM: %s\n", bom->type);
		bom_offset = bom->len;
	}
	
	aux = parser_init_result_file(txt_content + bom_offset, content_length - bom_offset);
	json_object_add(result, "details", aux);

	if (bom != NULL)
		json_object_add(aux, "BOM", json_string_new(bom->type));

	return(result);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * add_record(_INOUT JSON_VALUE *records)
{
	JSON_VALUE *rec = json_object_new(2);

	json_array_add(records, rec);

	return(rec);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_process(_IN JSON_VALUE *templ, _IN char *txt_content, _IN int content_length)
{
	JSON_VALUE *result = NULL, *records, *current_record, *details, *aux;
	char *buf = txt_content;
	char *p, *dupline;
	int line_n;

	result = parser_init_result(txt_content, content_length);
	records = json_object_find(result, "records");

	for (line_n = 1, p = buf, current_record = NULL ; (p - buf) < content_length ; line_n++)
	{
		dupline = runGetLine(&p);
		if (dupline == NULL)
			break;

		if (checkLineKey(templ, dupline))
		{
			current_record = add_record(records);
		}

		processLine(templ, result, current_record, dupline, line_n);
		if_free(dupline);
	}

	details = json_object_find(result, "details");
	aux = json_integer_new(json_array_length(records));
	json_object_add(details, "records_qty", aux);
	
	return(result);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
char * fixo2json(_IN char *json_template, _IN char *txt)
{
	JSON_VALUE *result, *temp;
	char *ret;

	if (modules == NULL)
		parser_init_modules();

	if ((json_template == NULL) || (txt == NULL))
		return(NULL);

	temp = json_parse_mem(json_template);
	if (temp == NULL)
		return(NULL);

	result = parser_process(temp, txt, strlen(txt));

	ret = json_serialize(result);

	json_value_free(temp);
	json_value_free(result);

	return(ret);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
_PRIVATE int csv2json_header(_INOUT char **fields, _IN char separador, _OUT char **header, _INOUT char **p)
{
	// Elimina linhas em branco no inicio do arquivo
	do
	{
		*header = runGetLine(p);
		if (**p == 0)
			return(-1);
	}
	while (strlen(*header) < 2);

	return(tokenizer(separador, *header, fields, MAX_CSV_FIELDS));
}
// ////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * parser_csv2json_in(char *csv, char separador)
{
	char *header, *line, *p;
	JSON_VALUE *json, *obj, *aux;
	char *fields[MAX_CSV_FIELDS + 1];
	char *tk[MAX_CSV_FIELDS + 1];
	int j, qty_fields, q;

	if (csv == NULL)
		return(NULL);

	p = csv;
	qty_fields = csv2json_header(fields, separador, &header, &p);
	if (qty_fields < 1)
		return(NULL);

	json = json_array_new(2);

	for (line = runGetLine(&p) ; line != NULL ; line = runGetLine(&p))
	{
		q = tokenizer(separador, line, tk, MAX_CSV_FIELDS);

		if (q >= qty_fields)
		{
			obj = json_object_new(2);
			for (j = 0 ; (j < qty_fields) && (j < q) ; j++)
			{
				aux = json_string_new(tk[j]);
				json_object_add(obj, fields[j], aux);
			}
        
			json_array_add(json, obj);
		}
		if_free(line);
	}

	if_free(line);
	if_free(header);

	return(json);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
_CALLBACK int parser_csv2json_check_keys_iter(JSON_VALUE *key, void *user_data)
{
	JSON_VALUE *record = (JSON_VALUE *) user_data;
	char *value = json_get_string(key);

	if (json_object_find(record, value) == NULL)
		return(1);

	return(0);
}
// ////////////////////////////////////////////////////////////
_PRIVATE int parser_csv2json_check_keys(JSON_VALUE *jsoncsv1, JSON_VALUE *templ)
{
	JSON_VALUE *keys, *record;
	int n;

	if (json_array_length(jsoncsv1) < 1)
		return(0);

	keys = json_object_find(templ, "keys");
	if (keys == NULL)
		return(1);

	record = json_array_index(jsoncsv1, 0);
	n = json_array_iter(keys, parser_csv2json_check_keys_iter, record);
	if (n < json_array_length(keys))
		return(0);

	return(1);
}
// ////////////////////////////////////////////////////////////
_CALLBACK int parser_csv2json_iter(JSON_VALUE *csvrecord, void *user_data)
{
	void **params = (void **) user_data;
	JSON_VALUE *templ = (JSON_VALUE *) params[0];
	JSON_VALUE *csvjson2 = (JSON_VALUE *) params[1];
	JSON_VALUE *records = json_object_find(csvjson2, "records");
	JSON_VALUE *fields = json_object_find(templ, "fields");
	char *csv_name, *name, *value;
	JSON_VALUE *record, *field, *jvalue;
	int i;

	if (records == NULL)
	{
		records = json_array_new(1);
		json_object_add(csvjson2, "records", records);
	}

	record = json_object_new(1);
	json_array_add(records, record);

	for (i = 0 ; i < json_array_length(fields) ; i++)
	{
		field = json_array_index(fields, i);
		csv_name = json_get_string(json_object_find(field, "csv_name"));
		name = json_get_string(json_object_find(field, "name"));

		value = if_malloc(10 * BUFFER_LEN);
		jvalue = json_object_find(csvrecord, csv_name);
		if (jvalue == NULL)
			value[0] = 0;
		else
			strncpy(value, json_get_string(jvalue), 10 * BUFFER_LEN);

		parser_apply_functions(value, NULL, record, NULL, NULL, field);
		json_object_add(record, name, json_string_new(value));

		if_free(value);
	}
	
	return(0);
}
// ////////////////////////////////////////////////////////////
_PUBLIC JSON_VALUE * parser_csv2json(char *csv, char separador, _IN JSON_VALUE *templ)
{
	char *list[] = {"source","parser_csv2json", NULL,NULL};
	JSON_VALUE *jsoncsv1, *jsoncsv2, *globals;
	int bom_offset = 0;
	void *params[2];
	BOM_CODE *bom;

	if (csv == NULL)
		return(NULL);

	bom = if_get_BOM(csv);
	if (bom != NULL)
	{
		fprintf(stderr, "BOM: %s\n", bom->type);
		bom_offset = bom->len;
	}

	jsoncsv1 = parser_csv2json_in(csv + bom_offset, separador);

	if ((templ == NULL) || (jsoncsv1 == NULL))
		return(jsoncsv1);

	if (!parser_csv2json_check_keys(jsoncsv1, templ))
	{
		verbose(stderr, "Chave nao confirmada.\n");
		return(NULL);
	}

	jsoncsv2 = json_object_new(2);

	globals = json_object_find(templ, "globals");
	if (globals == NULL)
		globals = json_object_new_list(list);
	else
		globals = json_clone(globals);

	json_object_add(jsoncsv2, "globals", globals);

	params[0] = templ;
	params[1] = jsoncsv2;
	json_array_iter(jsoncsv1, parser_csv2json_iter, params);

	json_value_free(jsoncsv1);

	return(jsoncsv2);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC char * csv2json(char *csv, char separador, _IN char *template)
{
	JSON_VALUE *templ, *result;
	char *aux;

	if (csv == NULL)
		return(if_strdup(""));

	if (template != NULL)
	{
		templ = json_parse_mem(template);
		if (templ == NULL)
			return(if_strdup(""));
	}
	else
		templ = NULL;

	if (modules == NULL)
		parser_init_modules();

	result = parser_csv2json(csv, separador, templ);
	aux = json_serialize(result);
	json_value_free(result);
	json_value_free(templ);

	return(aux);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
int json_add_fields_csv_add_iter(char *field, JSON_VALUE *val, void *user_data)
{
	void **params_out = (void **) user_data;
	JSON_VALUE *record = (JSON_VALUE *) params_out[0];
	char *key = (char *) params_out[1];

	if (strcmp(key, field) == 0)
		return(0);

	json_object_add(record, field, val);

	return(0);
}
// ////////////////////////////////////////////////////////////
int json_add_fields_csv_iter(JSON_VALUE *line, void *user_data)
{
	void **params_out = (void **) user_data;
	JSON_VALUE *record = (JSON_VALUE *) params_out[0];
	char *key = (char *) params_out[1];
	int *qty = (int *) params_out[2];
	char *rec_id, *csv_id;
	void *params_in[2];
	JSON_VALUE *aux;

	aux = json_object_find(record, key);
	if (aux == NULL)
		return(1);

	rec_id = json_get_string(aux);

	aux = json_object_find(line, key);
	if (aux == NULL)
		return(0);

	csv_id = json_get_string(aux);

	if (strcmp(rec_id, csv_id) == 0)
	{
		*qty += 1;
		params_in[0] = record;
		params_in[1] = params_out[1];
		json_object_iter(line, json_add_fields_csv_add_iter, params_in);
		return(1);
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
int json_add_fields_iter(JSON_VALUE *record, void *user_data)
{
	void **params_out = (void **) user_data;
	void *params_in[3];
	JSON_VALUE *csv_json = (JSON_VALUE *) params_out[0];

	params_in[0] = record;
	params_in[1] = params_out[1];
	params_in[2] = params_out[2];
	json_array_iter(csv_json, json_add_fields_csv_iter, params_in);

	return(0);
}
// ////////////////////////////////////////////////////////////
int json_result_add_fields(_INOUT JSON_VALUE *json_result, _IN JSON_VALUE *json_csv, char *key)
{
	JSON_VALUE *records;
	void *params[3];
	int qty = 0;

	if ((json_result == NULL) || (json_csv == NULL) || (key == NULL))
		return(-1);

	records = json_object_find(json_result, "records");
	if (records == NULL)
		return(-2);

	params[0] = json_csv;
	params[1] = key;
	params[2] = &qty;

	json_array_iter(records, json_add_fields_iter, params);

	return(qty);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * result_add_fields(char *result, char *csv, char separador, char *key)
{
	JSON_VALUE *res_json, *csv_json;
	char *ret;

	if ((result == NULL) || (csv == NULL))
		return(if_strdup(""));

	res_json = json_parse_mem(result);
	if (res_json == NULL)
		return(if_strdup(""));

	csv_json = parser_csv2json_in(csv, separador);
	if (csv_json == NULL)
		return(if_strdup(""));

	json_result_add_fields(res_json, csv_json, key);

	ret = json_serialize(res_json);
	json_value_free(csv_json);

	return(ret);
}
// ////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////
_PRIVATE int csv2json_generate_template_iter(char *key, JSON_VALUE *value, void *user_data)
{
	static int pos = 0;
	STRING_BUFFER *sb = (STRING_BUFFER *) user_data;
	char name[PATH_LEN];
	char *s, *d;

	if (key[0] == 0)
		return(0);

	for (s = key, d = name ; *s != 0 ; s++)
	{
		if (*s < 0)
			continue;

		if (*s == ' ')
		{
			*d++ = '_';
			continue;
		}

		*d++ = tolower(*s);
	}

	*d = 0;

	if (pos++)
		string_append(sb, ",\n");

	string_append(sb, "\t\t\t{\n\t\t\t\t'csv_name':'");
	string_append(sb, key);
	string_append(sb, "',\n\t\t\t\t'name':'");
	string_append(sb, name);
	string_append(sb, "',\n\t\t\t\t'formatting_function_calls':[{'function_definition_id':1}]\n\t\t\t}");
	
	return(0);
}
// ////////////////////////////////////////////////////////////
_PRIVATE int csv2json_generate_template_keys_iter(char *key, JSON_VALUE *value, void *user_data)
{
	static int pos = 0;
	STRING_BUFFER *sb = (STRING_BUFFER *) user_data;

	if (key[0] == 0)
		return(0);

	if (pos++)
		string_append(sb, ",");
	
	string_append(sb, "'");
	string_append(sb, key);
	string_append(sb, "'");

	return(0);
}
// ////////////////////////////////////////////////////////////
_PRIVATE char * csv2json_generate_template(JSON_VALUE *records)
{
	STRING_BUFFER *sb = string_new("{\n\t'globals':{'source':'parser_csv2json'},\n\t'keys':[");
	JSON_VALUE *record = json_array_index(records, 0);
	char *content;

	json_object_iter(record, csv2json_generate_template_keys_iter, sb);
	string_append(sb, "],\n\t'fields':[\n");
	json_object_iter(record, csv2json_generate_template_iter, sb);
	string_append(sb, "\n\t\t]\n}\n");

	content = if_strdup(string_get_text(sb));
	string_free(sb);

	return(content);
}
// ////////////////////////////////////////////////////////////



#ifdef STANDALONE

// ////////////////////////////////////////////////////////////
FILE * getFDIN(char *file)
{
	FILE *fd;

	if ((file[0] == '-') && (file[1] == 0))
		return(stdin);

	fd = fopen(file, "rb");
	if (fd == NULL)
		fprintf(stderr, "Falha ao tentar abrir: '%s'.\n", file);

	return(fd);
}
// ////////////////////////////////////////////////////////////



static IF_GETOPT opts[] = {
	{0, 'c', IF_GETOPT_TYPE_STRING, "config", "", 0, "Arquivo de configuracao JSON. (template)"},
	{0, 'f', IF_GETOPT_TYPE_STRING, "file", "", 0, "Arquivo de entrada (TXT de posicoes fixas)"},
	{0, 'j', IF_GETOPT_TYPE_NONE, "funcs", "", 0, "Gera JSON com as funcoes para Javascript."},
	{0, 'm', IF_GETOPT_TYPE_STRING, "mailing", "", 0, "Arquivo CSV com dados de destinos."},
	{0, 'v', IF_GETOPT_TYPE_STRING, "csv", "", 0, "Gera JSON a partir de um CSV."},
	{0, 't', IF_GETOPT_TYPE_NONE, "template", "", 0, "Gera template a partir de um CSV."},
	{0, 0, 0, 0, 0, 0, 0}
};


// ////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	JSON_VALUE *templ = NULL, *csv, *result = NULL;
	FILE *fd_in, *fd_config;
	char *content, *config_json;
	int content_length;
	int r;

	r = if_getopt(opts, argc, argv);
	if (	(r) ||
		(!if_getopt_isChecked(opts, "funcs") &&
		!if_getopt_isChecked(opts, "csv") &&
		(!if_getopt_isChecked(opts, "file") || !if_getopt_isChecked(opts, "config")))
	   )
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(opts);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -c <TEMPLATE> -f <TXT>\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n\tshell$ %s -c template.json -f arquivo.txt > resultado.json\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

	parser_init_modules();

	if (if_getopt_isChecked(opts, "funcs"))
	{
		content = show_json_funcs(modules);
		fprintf(stdout, "%s\n", content);
		if_free(content);
		return(0);
	}

	if (if_getopt_isChecked(opts, "config"))
	{
		config_json = if_getopt_getValue(opts, "config");
		fd_config = fopen(config_json, "r");
		if (fd_config == NULL)
		{
			fprintf(stderr, "Arquivo nao encontrado ou sem permissao: '%s'\n", config_json);
			return(1);
		}
		
		templ = json_parse_file(fd_config);
		if (templ == NULL)
		{
			fclose(fd_config);
			return(2);
		}
		fclose(fd_config);
	}

	if (if_getopt_isChecked(opts, "csv"))
	{
		if ((fd_in = getFDIN(if_getopt_getValue(opts, "csv"))) == NULL)
			return(1);

		content_length = if_get_content(fd_in, &content);
		fclose(fd_in);

		result = parser_csv2json(content, ';', templ);
		if_free(content);

		if (if_getopt_isChecked(opts, "template"))
			content = csv2json_generate_template(result);
		else
			content = json_serialize(result);

		fprintf(stdout, "%s", content);
		if_free(content);

		json_value_free(result);

		return(0);
	}

	if ((fd_in = getFDIN(if_getopt_getValue(opts, "file"))) == NULL)
		return(3);

	content_length = if_get_content(fd_in, &content);
	fclose(fd_in);

	result = parser_process(templ, content, content_length);
	if_free(content);
	
	if (if_getopt_isChecked(opts, "mailing"))
	{
		if ((fd_in = getFDIN(if_getopt_getValue(opts, "mailing"))) == NULL)
			return(3);
        
		if_get_content(fd_in, &content);
		fclose(fd_in);

		csv = parser_csv2json(content, ';', NULL);
		r = json_result_add_fields(result, csv, "id");
		fprintf(stderr, "%d registros sofreram atualizacao.\n", r);
	}

	printf("%s\n", json_serialize(result));

	json_value_free(templ);
	json_value_free(result);

	parser_finalize_modules();

	return(0);
}
// ////////////////////////////////////////////////////////////
#endif

