#include <ifractal.h>
#include <tunnel.h>

#include <glib.h>


extern IF_GETOPT configs[];

GSList *streams = NULL;
THREAD_STATE alive = TH_ALIVE;
gchar **ips_admin;

#define DESCRIPT 	"Reversao de conexoes TCP - client -> server"


// ///////////////////////////////////////////////////////////////////// //
TUNNEL_REVERSE_PAIR * reverse_new(char *port, char *name)
{
	TUNNEL_REVERSE_PAIR *pair = NULL;
	int ssock = openServerTCP(port);
	if (ssock < 1)
	{
		verbose(stderr, "Falha ao tentar abrir porta: %s\n", port);
		return(NULL);
	}

	pair = if_malloc(sizeof(TUNNEL_REVERSE_PAIR));
	pair->ssock = ssock;
	strncpy(pair->port, port, sizeof(pair->port));
	strncpy(pair->name, name, sizeof(pair->name));

	return(pair);
}
// ///////////////////////////////////////////////////////////////////// //
void reverse_free(TUNNEL_REVERSE_PAIR *pair)
{
	if (pair == NULL)
		return;

	if_closesocket(pair->ssock);
	if_free(pair);
}
// ///////////////////////////////////////////////////////////////////// //
int reverse_accept(TUNNEL_REVERSE_PAIR *pair)
{
	int sock, cli_len = sizeof(struct sockaddr);
	struct sockaddr cli_addr;
	char ip[INET_ADDRSTRLEN];

	sock = accept_timeout(pair->ssock, 0, &cli_addr, &cli_len);
	if (sock < 0)
	{
		verbose(stderr, "accept fail.\n");
		return(-1);
	}

	if (sock == 0)
		return(0);

	inet_ntop(AF_INET, &(((struct sockaddr_in *) &cli_addr)->sin_addr.s_addr), ip, sizeof(ip));
	if (tunnel_isAdmin(ip, ips_admin))
	{
		if (pair->admin_sock > 0)
		{
			verbose(stdout, "Reconecta admin: %s\n", pair->name);
			if_closesocket(pair->admin_sock);
		}

		pair->admin_sock = sock;
		strncpy(pair->admin_ip, ip, IP_LEN);
		verbose(stdout, "Accept Admin: %s.\n", pair->name);
	}
	else
	{
		if (pair->client_sock > 0)
		{
			verbose(stdout, "Reconecta client: %s (%s)\n", pair->name, ip);
			if_closesocket(pair->client_sock);
		}

		pair->client_sock = sock;
		strncpy(pair->client_ip, ip, IP_LEN);
		verbose(stdout, "Accept Client: %s (%s)\n", pair->name, ip);
	}

	return(sock);
}
// ///////////////////////////////////////////////////////////////////// //
int reverse_iter(TUNNEL_REVERSE_PAIR *pair, size_t *in, size_t *out)
{
	if ((pair->admin_sock <= 0) || (pair->client_sock <= 0))
	{
		reverse_accept(pair);
		return(0);
	}

	uint8_t bytes[BUFFER_LEN];
	int nin, nout;

	nout = read_timeout(pair->admin_sock, bytes, BUFFER_LEN, 0);
	if (nout > 0)
	{
		send_bytes(pair->client_sock, bytes, nout, 1);
	}
	else if (nout < 0)
	{
		//verbose(stdout, "Finaliza admin: %s\n", pair->name);
		if_closesocket(pair->admin_sock);
		pair->admin_sock = 0;
		return(-2);
	}
	*out += nout;

	nin = read_timeout(pair->client_sock, bytes, BUFFER_LEN, 0);
	if (nin > 0)
	{
		send_bytes(pair->admin_sock, bytes, nin, 1);
	}
	else if (nin < 0)
	{
		verbose(stdout, "Finaliza client: %s\n", pair->name);
		if_closesocket(pair->client_sock);
		pair->client_sock = 0;
		return(-3);
	}
	*in += nin;

	if ((nin + nout) > 0)
		verbose(stdout, "%s IN/OUT: %ld/%ld\n", pair->name, nin, nout);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int reverse_init_iter(JSON_VALUE *jstream, void *user_data)
{
	if (json_get_type(jstream) != JSON_OBJECT_TYPE)
		return(0);

	char *port = json_object_get_string(jstream, "port");
	char *name = json_object_get_string(jstream, "name");

	if ((port == NULL) || (port[0] == 0))
		return(0);

	TUNNEL_REVERSE_PAIR *pair = reverse_new(port, name);
	if (pair == NULL)
	{
		verbose(stdout, "Falha ao tentar iniciar port: '%s' - %s\n", port, name);
		return(0);
	}

	streams = g_slist_append(streams, pair);

	verbose(stdout, "%40s  -  Port: %s\n", name, port);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int reverse_init(char *filejson)
{
	char *admins = if_getopt_getValue(configs, "ADMINS");
	JSON_VALUE *jcsv, *jrecords, *jcsvtemp, *rjson = NULL;
	FILE *fd = fopen(filejson, "r");
	char *content = NULL;

	ips_admin = g_strsplit(admins, ",", 100);

	if (fd == NULL)
		return(1);

	fs_get_content(fd, &content);
	fclose(fd);

	if (g_str_has_suffix(filejson, ".csv"))
	{
		jcsvtemp = json_parse_mem(CSV_TEMPLATE);
		jcsv = parser_csv2json(content, ';', jcsvtemp);
		json_value_free(jcsvtemp);

		if (jcsv == NULL)
			return(2);

		jrecords = json_object_find(jcsv, "records");
		rjson = json_clone(jrecords);
		json_value_free(jcsv);
	}
	else if (g_str_has_suffix(filejson, ".json"))
		rjson = json_parse_mem(content);
	else
		return(3);

	if (rjson == NULL)
		return(4);

	if (json_get_type(rjson) != JSON_ARRAY_TYPE)
		return(5);

	json_array_iter(rjson, reverse_init_iter, NULL);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
void reverse_perform_callback(gpointer data, gpointer user_data)
{
	TUNNEL_REVERSE_PAIR *pair = (TUNNEL_REVERSE_PAIR *) data;
	void **ctx = (void **) user_data;
	size_t *in = (size_t *) ctx[0];
	size_t *out = (size_t *) ctx[1];

	reverse_iter(pair, in, out);
}
// ///////////////////////////////////////////////////////////////////// //
int reverse_run()
{
	size_t in = 0, out = 0;
	void *ctx[] = {&in, &out};

	while (alive == TH_ALIVE)
	{
		in = out = 0;
		g_slist_foreach(streams, reverse_perform_callback, ctx);
		if ((in + out) == 0)
		{
			if_sleep(10);
		}
		else if ((in + out) > 0)
		{
			verbose(stdout, "Global IN/OUT: %ld/%ld\n", in, out);
		}
		else
		{
			if_sleep(10);
		}
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void sighandler(int s)
{
	if (s == SIGPIPE)
		return;

	alive = TH_DEAD;
	fprintf(stdout, "Signal: %d\n", s);
}
// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char *argv[])
{
	char *config_ini = TUNNEL_INI;
	int r = 0;

	if (argc < 2)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n\t$ %s <TUNNEL_INI>\n", argv[0]);
		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\t$ %s -\n\n", argv[0]);
		return(1);
	}

	signal(SIGINT, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGQUIT, sighandler);
	signal(SIGPIPE, sighandler);
	signal(SIGHUP, sighandler);

	if (argv[1][0] != '-')
		config_ini = argv[1];

	if (if_getopt_ini(config_ini, configs) < 1)
	{
		verbose(stderr, "Falha ao tentar ler: %s\n", config_ini);
		return(1);
	}

	char *rjson = if_getopt_getValue(configs, "REVERSE_JSON");
	r = reverse_init(rjson);
	if (r != 0)
	{
		verbose(stderr, "Falha ao tentar processar: %s (%d)\n", rjson, r);
		return(1);
	}

	r = reverse_run();

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //


