#include <ifractal.h>
#include <ifdevice.h>

#include <zk.h>

int ZK_online_loop(_IN IFDEVICE *, _IN ZKHEADER *, _IN unsigned char *, IFDEVICE4J_online_callback, void *);


// ///////////////////////////////////////////////////////////////////// //
char * ZK_getCMD(uint16_t cmd)
{
	switch (cmd)
	{
		case CMD_CHECK: return("'Acorda' equipamento.");

		case CMD_CONNECT: return("Connection request");
		case CMD_EXIT: return("Disconnect");
		case CMD_ENABLEDEVICE: return("Enable machine in normal work state");
		case CMD_DISABLEDEVICE: return("Disable machine in work state, display “in the work ...” on LCD");
		case CMD_RESTART: return("Restart machine");
		case CMD_POWEROFF: return("Power off");
		case CMD_SLEEP: return("Enable machine in sleep");
		case CMD_RESUME: return("Awake sleeping machine (not support at present)");
		case CMD_CAPTUREFINGER: return("Capture fingerprint image");
		case CMD_TEST_TEMP: return("Test whether a fingerprint exists");
		case CMD_CAPTUREIMAGE: return("Capture all image");
		case CMD_REFRESHDATA: return("Refresh data in the machine");
		case CMD_REFRESHOPTION: return("Refresh configuration parameter");
		case CMD_TESTVOICE: return("Play voice");
		case CMD_GET_VERSION: return("Get firmware version");
		case CMD_CHANGE_SPEED: return("Change transmission speed");
		case CMD_AUTH: return("Connection authorization");
		case CMD_PREPARE_DATA: return("Prepare to transmit data");
		case CMD_DATA: return("send a data packet");
		case CMD_FREE_DATA: return("Free buffer memory");
		case CMD_REQUEST: return("Envia comando (coleta, lista, ?)");
		case CMD_REQUEST_DATA: return("Solicita pacote de dados");

		case CMD_DB_RRQ: return("Read a data in machine");
		case CMD_USER_WRQ: return("Upload user information (from PC to terminal)");
		case CMD_USERTEMP_RRQ: return("Read a fingerprint template or all data");
		case CMD_USERTEMP_WRQ: return("Upload a fingerprint template");
		case CMD_OPTIONS_RRQ: return("Read a configuration parameter in the machine");
		case CMD_OPTIONS_WRQ: return("Set machine configuration parameter");
		case CMD_ATTLOG_RRQ: return("Read all attendance record");
		case CMD_CLEAR_DATA: return("Clear data");

		case CMD_CLEAR_ATTLOG: return("Clear attendance log");
		case CMD_DELETE_USER: return("Delete some user");
		case CMD_DELETE_USERTEMP: return("delete a fingerprint template");
		case CMD_CLEAR_ADMIN: return("Clear administrator");

		case CMD_USERGRP_RRQ: return("Read user subgroup");
		case CMD_USERGRP_WRQ: return("Set user subgroup");
		case CMD_USERTZ_RRQ: return("Read user time zone setting");
		case CMD_USERTZ_WRQ: return("Write user time zone setting");
		case CMD_GRPTZ_RRQ: return("Read group time zone setting");
		case CMD_GRPTZ_WRQ: return("write group time zone setting");
		case CMD_TZ_RRQ: return("Read time zone setting");
		case CMD_TZ_WRQ: return("Write time zone setting");
		case CMD_ULG_RRQ: return("Read unlocking combination");
		case CMD_ULG_WRQ: return("Write unlocking combination");
		case CMD_UNLOCK: return("unlock");
		case CMD_CLEAR_ACC: return("Recover access control setting as default state");
		case CMD_CLEAR_OPLOG: return("Delete all attendance log in the machine");
		case CMD_OPLOG_RRQ: return("Read management record");
		case CMD_GET_FREE_SIZES: return("Get machine state, such as user record and so on");
		case CMD_ENABLE_CLOCK: return("Enable machine in normal work state");

		case CMD_STARTVERIFY: return("Enable machine in verification state");
		case CMD_STARTENROLL: return("Start to enroll a user, enable machine in enrolling user state");
		case CMD_CANCELCAPTURE: return("Enable machine in waiting for command CMD_STARTENROLL for detailed information.");
		case CMD_STATE_RRQ: return("Get machine state");
		case CMD_WRITE_LCD: return("Write LCD");
		case CMD_CLEAR_LCD: return("Clear LCD subtitle (clear screen)");
		case CMD_GET_PINWIDTH: return("Get user PIN length");

		case CMD_SMS_WRQ: return("Upload SMS");
		case CMD_SMS_RRQ: return("Download SMS");
		case CMD_DELETE_SMS: return("Delete SMS");
		case CMD_UDATA_WRQ: return("Set user SMS");
		case CMD_DELETE_UDATA: return("Delete user SMS");
		case CMD_DOORSTATE_RRQ: return("Get door state");

		case CMD_WRITE_MIFARE: return("Write Mifare card");
		case CMD_READ_MIFARE: return("Read Mifare card");
		case CMD_EMPTY_MIFARE: return("Clear Mifare card");
		case CMD_GET_TIME: return("Get machine time");
		case CMD_SET_TIME: return("Set machine time");

		case CMD_REG_EVENT: return("Register event");

		case CMD_ACK_OK: return("Returned value after successful execution");
		case CMD_ACK_ERROR: return("Returned value after failed execution");
		case CMD_ACK_DATA: return("Return value");
		case CMD_ACK_RETRY: return("Registered event occurred");
		case CMD_ACK_REPEAT: return("CMD_ACK_UNAUTH 2005 Unauthorized connection");
		case CMD_ACK_UNKNOWN: return("Unknown command");
		case CMD_ACK_ERROR_CMD: return("Command error");
		case CMD_ACK_ERROR_INIT: return("Not Initialized");
		case CMD_ACK_ERROR_DATA: return("Error data");
	}

	return("---");
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int ZK_openUDP(int port)
{
	struct sockaddr_in sock_addr;
	int sock;

	sock_addr.sin_family = AF_INET;
	sock_addr.sin_port = htons(port);
	sock_addr.sin_addr.s_addr = INADDR_ANY;
	memset(sock_addr.sin_zero, 0, sizeof(sock_addr.sin_zero));

#ifdef WIN32
	WSADATA wsaData;
	WSAStartup(MAKEWORD(2,2), &wsaData);

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) == SOCKET_ERROR)
	{
		verbose(stderr, "socket failed with error = %d\n", WSAGetLastError());
		return(-1);
	}

        unsigned long int nonBlockingMode = 1;
        ioctlsocket(sock, FIONBIO, &nonBlockingMode);

	if (bind(sock, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) == SOCKET_ERROR)
	{
		verbose(stderr, "bind failed with error = %d\n", WSAGetLastError());
		return(-1);
	}
#else
	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return(-1);

	bind(sock, (struct sockaddr *) &sock_addr, sizeof(sock_addr));
#endif

	return(sock);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int ZK_send_datagram(int sock, char *ip, int port, unsigned char *data, int len)
{
	struct sockaddr_in remote_addr;
	int n;

	remote_addr.sin_family = AF_INET;
	remote_addr.sin_port = htons(port);
	remote_addr.sin_addr.s_addr = inet_addr(ip);
	memset(remote_addr.sin_zero, 0, sizeof(remote_addr.sin_zero));
	
	n = sendto(sock, (char *)data, len, 0, (struct sockaddr *)&remote_addr, sizeof(struct sockaddr));
	if (n < len)
	{
#ifdef WIN32
		verbose(stderr, "sendto failed with error = %d\n", WSAGetLastError());
#else
		verbose(stderr, "sendto failed %d/%d\n", n, len);
#endif
	}

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int ZK_recv_datagram(int sock, unsigned char *data, size_t size, struct sockaddr_in *remote_addr)
{
	int n;

#ifdef WIN32
	int addr_len = sizeof(struct sockaddr_in);

	n = recvfrom(sock, (char *) data, size, MSG_PEEK, (struct sockaddr *)&remote_addr, &addr_len);
	if (n < 0)
	{
		//verbose(stderr, "recvfrom failed with error = %d\n", WSAGetLastError());
		return(n);
	}
	else if (n == 0)
		return(n);

	n = recvfrom(sock, (char *) data, size, 0, (struct sockaddr *) &remote_addr, &addr_len);
	if (n < 0)
	{
		verbose(stderr, "(2) recvfrom failed with error = %d\n", WSAGetLastError());
		return(n);
	}
#else
	unsigned int addr_len = sizeof(struct sockaddr_in);
	int flags = MSG_PEEK | MSG_DONTWAIT;

	n = recvfrom(sock, data, size, flags, (struct sockaddr *) remote_addr, &addr_len);
	if (n < 1)
		return(0);

	addr_len = sizeof(struct sockaddr_in);
	n = recvfrom(sock, data, size, 0, (struct sockaddr *) remote_addr, &addr_len);
#endif

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
uint16_t ZK_checksum(unsigned char *pack, size_t packlen)
{
	uint32_t sum = 0;
	int i;

	for (i = 0 ; i < packlen ; i += 2)
	{
		sum += pack[i];
		sum += (pack[i + 1] << 8);
	}

	return((((0xFFFF & sum) + (sum >> 16)) ^ 0xFFFF) & 0xFFFF);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_send(
	_IN IFDEVICE *reader, 
	_INOUT unsigned char *data, 
	_IN size_t len,
	_INOUT ZKHEADER *header)
{
	unsigned char *pack;
	size_t packlen;
	int n;

	packlen = sizeof(ZKHEADER) + len;
	pack = if_malloc(packlen + 1);

	header->checksum = 0;
	memcpy(pack, header, sizeof(ZKHEADER));

	if (len > 0)
		memcpy(pack + sizeof(ZKHEADER), data, len);

	((ZKHEADER *) pack)->checksum = ZK_checksum(pack, packlen);

	n = ZK_send_datagram(reader->sock, reader->host, atoi(reader->port), pack, packlen);
	if_free(pack);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_recv(
	_IN IFDEVICE *reader, 
	_INOUT unsigned char *data, 
	_INOUT ZKHEADER *header, 
	_IN int timeout)
{
	struct sockaddr_in remote_addr;
	time_t now = time(NULL);
	int n;

	memset(header, 0, sizeof(ZKHEADER));

	while ((n = ZK_recv_datagram(reader->sock, data, MAXPACKET, &remote_addr)) < 1)
	{
		if_sleep(50);
		if ((time(NULL) - now) > timeout)
			return(0);
	}

	memcpy(header, data, sizeof(ZKHEADER));

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_sendrecv(
	_IN IFDEVICE *reader, 
	_IN uint16_t cmd, 
	_INOUT unsigned char *data, 
	_IN size_t len, 
	_INOUT ZKHEADER *header)
{
	ZK *zk = (ZK *) reader;
	int timeout = 50;
	int n;

	if (data == NULL)
		return(-1);

	header->cmd = cmd;
	header->sessionid = zk->sessionid;
	header->replyid = zk->replyid++;

	n = ZK_send(reader, data, len, header);
	if (n < 0)
		return(n);

	n = ZK_recv(reader, data, header, timeout);
	if (n < 1)
	{
		verboseWARN(&(reader->log), "Erro: Equipamento nao responde...\n"); 
		return(-1);
	}

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
uint32_t ZK_encodeTime(struct tm *reg)
{
	uint32_t ano = 12 * 31 * 24 * 60 * 60;
	uint32_t dia = 24 * 60 * 60;
	uint32_t zktime = reg->tm_year * ano + ((reg->tm_mon + 1) * 31 + reg->tm_mday - 1) * dia 
				+ (reg->tm_hour * 60 + reg->tm_min) * 60 + reg->tm_sec;

	return(zktime);
}
// ///////////////////////////////////////////////////////////////////// //
time_t ZK_decodeTime(_IN uint32_t zktime)
{
	struct tm reg;
	memset(&reg, 0, sizeof(struct tm));

	for (reg.tm_year = 16 ; ZK_encodeTime(&reg) < zktime ; reg.tm_year++)
		;

	for (reg.tm_year--, reg.tm_mon = 1 ; ZK_encodeTime(&reg) < zktime ; reg.tm_mon++)
		;

	if (reg.tm_mon == 1)
	{
		reg.tm_mon = 13;
		reg.tm_year--;
	}

	for (reg.tm_mon--, reg.tm_mday = 1 ; ZK_encodeTime(&reg) < zktime ; reg.tm_mday++)
		;

	for (reg.tm_mday--, reg.tm_hour = 0 ; ZK_encodeTime(&reg) < zktime ; reg.tm_hour++)
		;

	for (reg.tm_hour--, reg.tm_min = 0 ; ZK_encodeTime(&reg) < zktime ; reg.tm_min++)
		;

	reg.tm_min--;
	reg.tm_sec = zktime - ZK_encodeTime(&reg);
	reg.tm_isdst = -1;

	reg.tm_year += 100;
	reg.tm_mon += 1;

	return(mktime(&reg));
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int ZK_open(IFDEVICE *reader, int wakeup)
{
	ZK *zk = (ZK *) reader;
	unsigned char data[MAXPACKET];
	int r, local_port = ZK_START_PORT;
	ZKHEADER header;

	zk->sessionid = 0;
	zk->replyid = 0;

	while ((reader->sock = ZK_openUDP(local_port)) < 0)
		local_port++;

	verboseDEBUG(&(reader->log), "Inicia UDP PORT: %d\n", local_port);

	for (r = 0 ; r < wakeup ; r++)
	{
		header.cmd = CMD_CHECK;
		header.sessionid = 0xE000;
		header.replyid = 0x5526;
		ZK_send(reader, NULL, 0, &header);
		if_sleep(4000);
	}

	r = ZK_sendrecv(reader, CMD_CONNECT, data, 0, &header);
	if (r < 0)
	{
		if_closesocket(reader->sock);
		return(r);
	}

	zk->sessionid = header.sessionid;

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_close(IFDEVICE *reader)
{
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	int r;

	r = ZK_sendrecv(reader, CMD_EXIT, data, 0, &header);
	if_closesocket(reader->sock);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
time_t ZK_getTime(IFDEVICE *reader)
{
	time_t unixtime;
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	uint32_t zktime;

	ZK_sendrecv(reader, CMD_GET_TIME, data, 0, &header);
	memcpy(&zktime, data + sizeof(ZKHEADER), sizeof(zktime));
	unixtime = ZK_decodeTime(zktime);

	return(unixtime);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_setTime(IFDEVICE *reader, int timediff)
{
	unsigned char data[MAXPACKET];
	time_t now = time(NULL);
	ZKHEADER header;
	uint32_t zktime;
	struct tm *dt;
	int r = 0;

	now += 3600 * timediff;
	dt = localtime(&now);
	dt->tm_year -= 100;
	dt->tm_mon -= 1;

	zktime = ZK_encodeTime(dt);
	memcpy(data, &zktime, sizeof(zktime));

	r = ZK_sendrecv(reader, CMD_SET_TIME, data, sizeof(zktime), &header);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int ZK_getInfo(IFDEVICE *reader, JSON_VALUE *info)
{
	char *params[] = {
		"DeviceID",
		"~DeviceName",
		"~Platform",
		"~ZKFPVersion",
		"~ProductTime",
		"~SerialNumber",
		"ExtendOPLog",
		"WorkCode",
		"~UserExtFmt",
		"~ExtendFmt",
		"FaceFunOn",
		"~IsOnlyRFMachine",
		NULL
	};
	char *states[] = {
		"",
		"",
		"",
		"",
		"users",
		"",
		"templates",
		"",
		"events",
		"",
		"", //"Super Registros",
		"",
		"",
		"",
		"Total_Digitais",
		"Total_Usuarios",
		"Total_Marcacoes",
		"",
		"",
		"",
		"Faces",
		"",
		"Total_Faces",
		NULL
	};
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	uint32_t value;
	int r, i, len;
	char *strdata;

	for (i = 0 ; params[i] != NULL ; i++)
	{
		len = strlen(params[i]) + 1;
		memcpy(data, params[i], len);
		r = ZK_sendrecv(reader, CMD_OPTIONS_RRQ, data, len, &header);
		strdata = (char *) (data + sizeof(ZKHEADER));

		// Corta caracteres invalidos

		int k;
		for (k = 0 ; strdata[k] != 0 ; k++)
		{
			if ((strdata[k] < ' ') || (strdata[k] > '~'))
			{
				strdata[k] = 0;
				break;
			}
		}

		json_object_add(info, params[i], json_string_new(strdata + strlen(params[i]) + 1));

		// TODO - verificar modelo
		((ZK *)reader)->model = ZK_FACE_MODEL;

		if ((strcmp(params[i], "~DeviceName") == 0) && (strstr(strdata, "PassFace") > 0))
		{
			((ZK *)reader)->model = ZK_FACE_MODEL;
			verboseDEBUG(&(reader->log), "Biometria Facial.\n");
		}
		else
			((ZK *)reader)->model = ZK_PLAIN_MODEL;
	}

	r = ZK_sendrecv(reader, CMD_GET_VERSION, data, 0, &header);
	strdata = (char *) (data + sizeof(ZKHEADER));
	json_object_add(info, "firmware", json_string_new(strdata));

	len = ZK_sendrecv(reader, CMD_GET_FREE_SIZES, data, 0, &header);
	for (i = 0 ; states[i] != NULL ; i++)
	{
		if (states[i][0] == 0)
			continue;

		r = sizeof(ZKHEADER) + i * sizeof(value);
		if (r > len)
			break;

		memcpy(&value, data + r, sizeof(value));

		if ((value < 0) || (value > 2000000))
			break;

		json_object_add(info, states[i], json_integer_new(value));
		verboseDEBUG(&(reader->log), "%s: %d\n", states[i], value);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void ZK_coleta_plain_in(JSON_VALUE *offs, ZKLOG *log)
{
	char datahora[PATH_LEN];
	JSON_VALUE *evt;
	time_t unixtime;
	struct tm *reg;

	unixtime = ZK_decodeTime(log->zktime);
	reg = localtime(&unixtime);

	evt = json_object_new(0);
	json_array_add(offs, evt);

	snprintf(datahora, sizeof(datahora), "%d-%02d-%02d %02d:%02d:%02d", 
		reg->tm_year + 1900, reg->tm_mon + 1, reg->tm_mday, 
		reg->tm_hour, reg->tm_min, reg->tm_sec);

	json_object_add(evt, "datahora", json_string_new(datahora));

	json_object_add(evt, "cracha", json_integer_new(log->pin));
	json_object_add(evt, "nro", json_integer_new(log->pin));
}
// ///////////////////////////////////////////////////////////////////// //
void ZK_coleta_face_in(JSON_VALUE *offs, ZKFACELOG *log)
{
	char datahora[PATH_LEN];
	JSON_VALUE *evt;
	time_t unixtime;
	struct tm *reg;

	// Ajusta alinhamento
	log->zktime <<= 8;
	log->zktime += log->status_padding[1];

	unixtime = ZK_decodeTime(log->zktime);
	reg = localtime(&unixtime);

	evt = json_object_new(0);
	json_array_add(offs, evt);

	json_object_add(evt, "nro", json_string_new(log->matricula));

	snprintf(datahora, sizeof(datahora), "%d-%02d-%02d %02d:%02d:%02d", 
		reg->tm_year + 1900, reg->tm_mon + 1, reg->tm_mday, 
		reg->tm_hour, reg->tm_min, reg->tm_sec);

	json_object_add(evt, "datahora", json_string_new(datahora));
	json_object_add(evt, "cracha", json_string_new(log->matricula));
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_coleta_in(IFDEVICE *reader, JSON_VALUE *offs, unsigned char *data, int len)
{
	int total_regs, i, loglen;
	unsigned char *ptr;
	uint32_t total;
	union
	{
		ZKLOG plain;
		ZKFACELOG face;
	} log;

	if (((ZK *) reader)->model == ZK_PLAIN_MODEL)
		loglen = sizeof(ZKLOG);
	else
		loglen = sizeof(ZKFACELOG);

	memcpy(&total, data, sizeof(total));
	total_regs = total / loglen;
	if (total > 0)
		verboseINFO(&(reader->log), "Quantidade total: %d\n", total_regs);

	ptr = data + sizeof(total);
	for (i = 1 ; (ptr - data) < len ; ptr += loglen, i++)
	{
		memcpy(&log, ptr, loglen);

		if (((ZK *) reader)->model == ZK_PLAIN_MODEL)
			ZK_coleta_plain_in(offs, (ZKLOG *) &log);
		else
			ZK_coleta_face_in(offs, (ZKFACELOG *) &log);
	}

	return(total_regs);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_coleta_extra(IFDEVICE *reader, JSON_VALUE *offs, unsigned char *data, int len)
{
	struct { uint32_t total; uint32_t size; } ZKsize;
	unsigned char *buflogs, *p;
	int loglen, regs, r = -1;
	ZKHEADER header;
	uint32_t total;

	memcpy(&total, data + 1, sizeof(total));

	if (((ZK *) reader)->model == ZK_PLAIN_MODEL)
		loglen = sizeof(ZKLOG);
	else
		loglen = sizeof(ZKFACELOG);
	
	regs = (total - sizeof(uint32_t)) / loglen;
	verboseDEBUG(&(reader->log), "Coleta %d registros.\n", regs);

	ZKsize.total = 0;
	ZKsize.size = total;
	memcpy(data, &ZKsize, sizeof(ZKsize));
	len = ZK_sendrecv(reader, CMD_REQUEST_DATA, data, sizeof(ZKsize), &header);
	if (len < 1)
		return(-2);

	if ((header.cmd != CMD_PREPARE_DATA) || (len != (sizeof(ZKHEADER) + sizeof(ZKsize))))
	{
		verboseWARN(&(reader->log), "Falha na coleta: comando invalido.\n");
		return(-3);
	}

	memcpy(&ZKsize, data + sizeof(ZKHEADER), sizeof(ZKsize));
	verboseDEBUG(&(reader->log), "Recebe primeiro lote: %d/%d.\n", ZKsize.size, ZKsize.total);

	buflogs = if_malloc(total);
	for (p = buflogs ; (p - buflogs) < total; p += len)
	{
		len = ZK_recv(reader, data, &header, 2) - sizeof(ZKHEADER);
		if (header.cmd != CMD_DATA)
		{
			verboseWARN(&(reader->log), "Falha na coleta: retorno inesperado.\n");
			goto ZK_coleta_extra_err;
		}

		memcpy(p, data + sizeof(ZKHEADER), len);
	}

	ZK_recv(reader, data, &header, 2);
	if (header.cmd == CMD_ACK_OK)
	{
		ZK_sendrecv(reader, CMD_FREE_DATA, data, 0, &header);
		r = ZK_coleta_in(reader, offs, buflogs, total);
	}
	else
		verboseWARN(&(reader->log), "(%d) Falha na coleta: dados inconsistentes.\n", reader->nro);

ZK_coleta_extra_err:
	if_free(buflogs);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_coleta(IFDEVICE *reader, JSON_VALUE *offs)
{
	unsigned char data[MAXPACKET] = {0x01,0x0D};
	int len, r, loglen;
	ZKHEADER header;

	len = ZK_sendrecv(reader, CMD_REQUEST, data, 11, &header) - sizeof(ZKHEADER);
	if (len < 0)
	{
		verboseWARN(&(reader->log), "Falha ao tentar realizar coleta.\n");
		return(-1);
	}

	if (header.cmd == CMD_ACK_ERROR)
		return(0);

	if (((ZK *) reader)->model == ZK_PLAIN_MODEL)
		loglen = sizeof(ZKLOG);
	else
		loglen = sizeof(ZKFACELOG);

	if (((len - sizeof(uint32_t)) % loglen) != 0)
		r = ZK_coleta_extra(reader, offs, data + sizeof(ZKHEADER), len - sizeof(ZKHEADER));
	else
		r = ZK_coleta_in(reader, offs, data + sizeof(ZKHEADER), len - sizeof(ZKHEADER));

	if (r)
		ZK_sendrecv(reader, CMD_CLEAR_ATTLOG, data, 0, &header);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int ZK_deleteUser(IFDEVICE *reader, ZKUSER *user)
{
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	int len;

	memcpy(data, user, sizeof(ZKUSER));
	len = ZK_sendrecv(reader, CMD_DELETE_USER, data, sizeof(ZKUSER), &header);
	if (len < 8)
	{
		verboseWARN(&(reader->log), "Falha ao tentar deletar usuario.\n");
		return(-1);
	}

	len = ZK_sendrecv(reader, CMD_FREE_DATA, data, 0, &header);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_sendUser(IFDEVICE *reader, ZKUSER *user)
{
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	int len;

	memcpy(data, user, sizeof(ZKUSER));
	len = ZK_sendrecv(reader, CMD_USER_WRQ, data, sizeof(ZKUSER), &header);
	if ((len < 8) || (header.cmd != CMD_ACK_OK))
	{
		verboseWARN(&(reader->log), "Falha ao tentar enviar usuario.\n");
		return(-1);
	}

	len = ZK_sendrecv(reader, CMD_FREE_DATA, data, 0, &header);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_sendFaceUser(IFDEVICE *reader, unsigned char *data)
{
	ZKHEADER header;
	int len;

	len = ZK_sendrecv(reader, CMD_USER_WRQ, data, ZKFACEUSER_PACK_LEN, &header);
	if (len < 8)
	{
		verboseWARN(&(reader->log), "Falha ao tentar enviar usuario.\n");
		return(-1);
	}

	len = ZK_sendrecv(reader, CMD_FREE_DATA, data, 0, &header);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int ZK_load_list_face_user(IFDEVICE *reader, JSON_VALUE *juser)
{
	ZKFACEUSER user;
	unsigned char data[ZKFACEUSER_PACK_LEN];
	char *aux;
	int r;

	memset(data, 0, ZKFACEUSER_PACK_LEN);

	aux = json_object_get_string(juser, "codigo");
	user.pin = atoi(aux);		// indice

	aux = json_object_get_string(juser, "nome");
	snprintf(user.name, sizeof(user.name), "%s", aux);

	aux = json_object_get_string(juser, "senha");
	snprintf(user.pass, sizeof(user.pass), "%s", aux);

	aux = json_object_get_string(juser, "cracha");
	snprintf(user.matricula, sizeof(user.matricula), "%s", aux);
	user.card = atoi(aux);

	memcpy(data + ZKFACEUSER_OFFSET_PIN, &(user.pin), sizeof(user.pin));
	memcpy(data + ZKFACEUSER_OFFSET_PASS, user.pass, sizeof(user.pass));
	memcpy(data + ZKFACEUSER_OFFSET_NAME, user.name, sizeof(user.name));
	memcpy(data + ZKFACEUSER_OFFSET_CARD, &(user.card), sizeof(user.card));
	memcpy(data + ZKFACEUSER_OFFSET_MATRICULA, user.matricula, sizeof(user.matricula));

	r = ZK_sendFaceUser(reader, data);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int ZK_load_list_iter_user_cartao(JSON_VALUE *jcartao, void *user_data) {

	ZKUSER user;
	char *aux;
	int n;
	void **params = (void **) user_data;
	IFDEVICE *reader = (IFDEVICE *) params[0];
	JSON_VALUE *juser = (JSON_VALUE *) params[1];
	int *r = (int *) params[2];

	memset(&user, 0, sizeof(ZKUSER));

	aux = json_object_get_string(jcartao, "codigo");
	user.pin = (uint16_t) atoi(aux);	// indice
	//user.privilege = ZK_ORDINARY_USER;
	user.privilege = json_object_get_int(juser, "privilege");

	user.group = 1;				// grupo default
	user.unknown1 = 0xA4FFFFFF;
	user.unknown2 = 0;

	aux = json_object_get_string(juser, "nome");
	snprintf(user.name, sizeof(user.name), "%s", aux);

	aux = json_object_get_string(jcartao, "nro");
	n = strlen(aux);
	user.rfid = atoi(aux);
	user.id = user.rfid;

	if (n > 5)
	{
		user.rfid = atoi(aux + (n - 5));
		aux[n - 5] = 0;
		n = atoi(aux);
		((unsigned char *) &user.rfid)[2] = (unsigned char) (n & 0xFF);
	}

	if (strcmp(json_object_get_string(juser, "tipo"), "excluir") == 0)
		*r += ZK_deleteUser(reader, &user);
	else
		*r += ZK_sendUser(reader, &user);

	return(0);

}

// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int ZK_load_list_user(IFDEVICE *reader, JSON_VALUE *juser)
{
	int r = 0;
	void *params[] = {reader, juser, &r};

	JSON_VALUE* jnro_cartao = json_object_find(juser, "nro_cartao");

    if ((jnro_cartao != NULL) && (json_get_type(jnro_cartao) == JSON_ARRAY_TYPE))
		json_array_iter(jnro_cartao, ZK_load_list_iter_user_cartao, params);
	else
	{
		json_object_add(juser, "nro", json_object_find(juser, "cracha"));
		ZK_load_list_iter_user_cartao(juser, params);
	}
		
	return 0;
}
// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int ZK_load_list_iter(JSON_VALUE *juser, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *reader = (IFDEVICE *) params[0];
	JSON_VALUE *res = (JSON_VALUE *) params[1];
	JSON_VALUE *evt;
	char msg[PATH_LEN];
	char *cracha = json_object_get_string(juser, "cracha");
	char *cod = json_object_get_string(juser, "cod");
	char *codigo = json_object_get_string(juser, "codigo");
	time_t now = time(NULL);
	struct tm *dt;
	int r;

	if (((ZK *) reader)->model == ZK_PLAIN_MODEL)
		r = ZK_load_list_user(reader, juser);
	else if (((ZK *) reader)->model == ZK_FACE_MODEL)
		r = ZK_load_list_face_user(reader, juser);
	else
		return(0);

	evt = json_object_new(1);
	json_object_add(evt, "cracha", json_string_new(cracha));
	json_object_add(evt, "cod_error", json_integer_new(r));

	if (r == 0)
		json_object_add(evt, "msg", json_string_new("OK"));
	else
		json_object_add(evt, "msg", json_string_new("Falha ao tentar incluir/excluir."));

	dt = localtime(&now);
	snprintf(msg, sizeof(msg), "%04d-%02d-%02d %02d:%02d:%02d", 
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday, dt->tm_hour, dt->tm_min, dt->tm_sec);

	json_object_add(evt, "datahora", json_string_new(msg));
	json_object_add(evt, "cod", json_string_new(cod));
	json_object_add(evt, "codigo", json_string_new(codigo));
	
	json_array_add(res, evt);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void ZK_user2json(JSON_VALUE *res, ZKUSER *usr)
{
	JSON_VALUE *jusr = json_object_new(1);
	unsigned char *p;
	uint32_t rfid;
	char name[9];
	char cracha[20];
	
	memcpy(name, usr->name, 8);
	name[8] = 0;

	for (p = (unsigned char *) name ; *p != 0 ; p++)
		if ((*p < '0') || (*p > 'z') || ((*p > 'Z') && (*p < 'a')))
			*p = ' ';

	rfid = usr->rfid & 0xFFFF;
	rfid += (usr->rfid >> 16) * 100000;
	snprintf(cracha, sizeof(cracha), "%d", rfid);

	json_array_add(res, jusr);
	json_object_add(jusr, "codigo", json_integer_new(usr->pin));
	json_object_add(jusr, "cracha", json_string_new(cracha));
	json_object_add(jusr, "nome", json_string_new(name));
	json_object_add(jusr, "privilege", json_integer_new(usr->privilege));
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_user_in(IFDEVICE *dev, JSON_VALUE *res, unsigned char *data, int len)
{
	int total_regs, i, usrlen;
	unsigned char *ptr;
	uint32_t total;
	union
	{
		ZKUSER plain;
		ZKFACEUSER face;
	} usr;

	if (((ZK *) dev)->model == ZK_PLAIN_MODEL)
		usrlen = sizeof(ZKUSER);
	else
		usrlen = sizeof(ZKFACEUSER);

	memcpy(&total, data, sizeof(total));
	total_regs = total / usrlen;
	if (total > 0)
		verboseINFO(&(dev->log), "Total usuarios: %d\n", total_regs);

	ptr = data + sizeof(total);
	for (i = 1 ; (ptr - data) < len ; ptr += usrlen, i++)
	{
		memcpy(&usr, ptr, usrlen);

		if (((ZK *) dev)->model == ZK_PLAIN_MODEL)
			ZK_user2json(res, (ZKUSER *) &usr);
		//else
		//	ZK_faceuser2json(res, (ZKFACEUSER *) &usr);
	}

	return(total_regs);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_user_extra(IFDEVICE *dev, JSON_VALUE *offs, unsigned char *data, int len)
{
	struct { uint32_t total; uint32_t size; } ZKsize;
	unsigned char *bufusrs, *p;
	int r = -1;
	ZKHEADER header;
	uint32_t total;

	memcpy(&total, data + 1, sizeof(total));
	
	ZKsize.total = 0;
	ZKsize.size = total;
	memcpy(data, &ZKsize, sizeof(ZKsize));
	len = ZK_sendrecv(dev, CMD_REQUEST_DATA, data, sizeof(ZKsize), &header);
	if (len < 1)
		return(-2);

	if ((header.cmd != CMD_PREPARE_DATA) || (len != (sizeof(ZKHEADER) + sizeof(ZKsize))))
	{
		verboseWARN(&(dev->log), "Falha na coleta: comando invalido.\n");
		return(-3);
	}

	memcpy(&ZKsize, data + sizeof(ZKHEADER), sizeof(ZKsize));
	verboseDEBUG(&(dev->log), "Recebe primeiro lote de usuarios: %d/%d bytes.\n", ZKsize.size, ZKsize.total);

	bufusrs = if_malloc(total);
	for (p = bufusrs ; (p - bufusrs) < total; p += len)
	{
		len = ZK_recv(dev, data, &header, 2) - sizeof(ZKHEADER);
		if (header.cmd != CMD_DATA)
		{
			verboseWARN(&(dev->log), "Falha ao ler usuarios: retorno inesperado.\n");
			goto ZK_user_extra_err;
		}

		memcpy(p, data + sizeof(ZKHEADER), len);
	}

	ZK_recv(dev, data, &header, 2);
	if (header.cmd == CMD_ACK_OK)
	{
		r = ZK_user_in(dev, offs, bufusrs, total);
	}
	else
		verboseWARN(&(dev->log), "(%d) Falha ao ler usuarios: dados inconsistentes.\n", dev->nro);

ZK_user_extra_err:
	if_free(bufusrs);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //



// Implementacao interface Java //////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_getInfo4J(IFDEVICE4J *dev, long sock)
{
	JSON_VALUE *info = json_object_new(1);

	if (ZK_open((IFDEVICE *) dev, 1) < 0)
		return(NULL);

	ZK_getInfo((IFDEVICE *) dev, info);

	ZK_close((IFDEVICE *) dev);

	return(info);
}
// ///////////////////////////////////////////////////////////////////// //
time_t ZK_getTime4J(IFDEVICE4J *dev, long sock)
{
	time_t dt = 0;

	if (ZK_open((IFDEVICE *) dev, 4) < 0)
		return(0);

	dt = ZK_getTime((IFDEVICE *) dev);

	ZK_close((IFDEVICE *) dev);

	return(dt);
}
// ///////////////////////////////////////////////////////////////////// //
intptr_t ZK_setTime4J(IFDEVICE4J *dev, intptr_t diff)
{
	int r = 0;

	if (ZK_open((IFDEVICE *) dev, 0) < 0)
		return(0);

	r = ZK_setTime((IFDEVICE *) dev, diff);

	ZK_close((IFDEVICE *) dev);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_sendUsers4J(IFDEVICE4J *dev4j, JSON_VALUE *users)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	JSON_VALUE *res = json_array_new(1);
	void *params[] = {dev, res};
	unsigned char data[MAXPACKET];
	ZKHEADER header;

	ZK_open(dev, 0);
	ZK_sendrecv(dev, CMD_DISABLEDEVICE, data, 0, &header);
	json_array_iter(users, ZK_load_list_iter, params);
	ZK_sendrecv(dev, CMD_REFRESHDATA, data, 0, &header);
	ZK_sendrecv(dev, CMD_ENABLEDEVICE, data, 0, &header);
	ZK_close(dev);

	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_getUsers4J(IFDEVICE4J *dev4j, intptr_t none)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	JSON_VALUE *res;
	int len, n;

	if (((ZK *) dev)->model != ZK_PLAIN_MODEL)
		return(NULL);

	ZK_open(dev, 0);
	n = ZK_sendrecv(dev, CMD_DISABLEDEVICE, data, 0, &header);
	if (n < 0)
		goto ZK_getUsers4J_err;

	data[0] = 0x01;
	data[1] = 0x09;
	data[2] = 0x00;
	data[3] = 0x05;
	len = ZK_sendrecv(dev, CMD_REQUEST, data, 11, &header) - sizeof(ZKHEADER);
	if (len < 0)
	{
		verboseWARN(&(dev->log), "Falha ao tentar recuperar cadastro.\n");
		goto ZK_getUsers4J_end;
	}

	if (header.cmd == CMD_ACK_ERROR)
		goto ZK_getUsers4J_end;

	res = json_array_new(1);

	if (((len - sizeof(uint32_t)) % sizeof(ZKUSER)) != 0)
		n = ZK_user_extra(dev, res, data + sizeof(ZKHEADER), len - sizeof(ZKHEADER));
	else
		n = ZK_user_in(dev, res, data + sizeof(ZKHEADER), len - sizeof(ZKHEADER));

ZK_getUsers4J_end:
	n = ZK_sendrecv(dev, CMD_ENABLEDEVICE, data, 0, &header);
ZK_getUsers4J_err:
	ZK_close(dev);

	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_sendBio4J(IFDEVICE4J *dev, JSON_VALUE *users)
{
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_getBio4J(IFDEVICE4J *dev, JSON_VALUE *users)
{
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * ZK_getEvents4J(IFDEVICE4J *dev, intptr_t nsr)
{
	JSON_VALUE *offs;

	if (ZK_open((IFDEVICE *) dev, 0) < 0)
		return(NULL);

	offs = json_array_new(1);

	ZK_coleta((IFDEVICE *) dev, offs);

	ZK_close((IFDEVICE *) dev);

	return(offs);
}
// ///////////////////////////////////////////////////////////////////// //
void ZK_free4J(IFDEVICE4J *dev)
{
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_online(IFDEVICE4J *dev, IFDEVICE4J_online_callback callback, void *context)
{
/*
	unsigned char data[MAXPACKET];
	char regevt[] = {0xFF, 0xFF, 0x0, 0x0};
	ZKHEADER header;
	int n;

	do
	{
		ZK_open(dev, 0);
		memcpy(data, regevt, sizeof(regevt));
		ZK_sendrecv(dev, CMD_REG_EVENT, data, sizeof(regevt), &header);
		n = ZK_online_loop(dev, &header, data, callback, context);
		ZK_close(dev);

		//ZK_idle(reader);
	} while (n >= 0);

	return(0);
*/
	return(5);
}
// ///////////////////////////////////////////////////////////////////// //
void ZK_init(IFDEVICE4J *dev)
{
	dev->getInfo = (IFDEVICE4J_getInfo) ZK_getInfo4J;
	dev->getTime = (IFDEVICE4J_getTime) ZK_getTime4J;
	dev->setTime = ZK_setTime4J;
	dev->sendUsers = ZK_sendUsers4J;
	dev->getUsers = ZK_getUsers4J;
	dev->sendBio = ZK_sendBio4J;
	dev->getBio = ZK_getBio4J;
	dev->getEvents = ZK_getEvents4J;
	dev->free = ZK_free4J;
	dev->online = ZK_online;
}
// ///////////////////////////////////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void ZK_ZKONLINE2ZKONLINE(_INOUT ZKONLINE *reg, _IN ZKONLINEFACE *zkon)
{
	memset(reg, 0, sizeof(ZKONLINE));

	// TODO - capturar os dados do online
	//snprintf(reg->matricula, sizeof(reg->matricula), "%s", zkon->matricula);

	reg->year = zkon->year;
	reg->mon = zkon->mon;
	reg->day = zkon->day;
	reg->hour = zkon->hour;
	reg->min = zkon->min;
	reg->sec = zkon->sec % 60;
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_online_loop(IFDEVICE *dev, ZKHEADER *header, unsigned char *data, IFDEVICE4J_online_callback callback, void *context)
{
	ZK *zk = (ZK *) dev;
	char token[] = "DeviceID";
	ZKONLINE reg;
	ZKONLINEFACE zkon;
	int n, i;
	char *sensor = "0", *channel = "0";
	char aux[PATH_LEN];
	char *args[] = {aux};

	do
	{
		n = ZK_recv(dev, data, header, 2);
		if (n < 0)
			break;

		if (n < 1)
		{
			memcpy(data, token, sizeof(token));
			n = ZK_sendrecv(dev, CMD_OPTIONS_RRQ, data, sizeof(token), header);
		}

		if (header->cmd == CMD_ACK_OK)
			continue;

		if (header->cmd != CMD_REG_EVENT)
		{
			verboseFATAL(&(dev->log), "Erro: Retorno nao implementado.\n");

			break;
		}

		// TODO - Ajustar o sistema para tratar o codigo ONLINE
		header->cmd = CMD_ACK_OK;
		header->sessionid = zk->sessionid;
		header->replyid = 0;
		ZK_send(dev, NULL, 0, header);

		if (n < (sizeof(ZKHEADER) + sizeof(ZKONLINEFACE)))
			continue;

		memcpy(&zkon, data + sizeof(ZKHEADER), sizeof(ZKONLINEFACE));
		verboseDEBUG(&(dev->log), 
			"%s - %02d/%02d/20%02d %02d:%02d:%02d\n", 
			zkon.matricula, 
			zkon.day, zkon.mon, zkon.day, zkon.hour, zkon.min, zkon.sec);

		ZK_ZKONLINE2ZKONLINE(&reg, &zkon);

		if (callback != NULL)
		{
			snprintf(aux, sizeof(aux), "%s;%s;%s", zkon.matricula, sensor, channel);
			verboseDEBUG(&(dev->log), "Solicita acesso: %s\n", zkon.matricula);
			callback(1, args, context, "getUserOnline");
		}
	} while ((n >= 0) && ((i++ % 150) != 0));

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //


#ifdef STANDALONE


IF_GETOPT configs[] = {
	{0, 'h', IF_GETOPT_TYPE_STRING, "host", "", 0, "IP do equipamento."},
	{0, 'p', IF_GETOPT_TYPE_STRING, "port", "4370", 0, "Porta (UDP) do equipamento."},
	{0, 'u', IF_GETOPT_TYPE_STRING, "update", "", 0, "Nome do arquivo JSON com a lista de usuarios."},
	{0, 'r', IF_GETOPT_TYPE_BOOLEAN, "read", "", 0, "Recupera lista de usuarios."},
	{0, 'x', IF_GETOPT_TYPE_BOOLEAN, "delete-all", "", 0, "Exclui todos os usuarios."},
	{0, 'e', IF_GETOPT_TYPE_BOOLEAN, "log", "", 0, "Recupera eventos."},
	{0, 'd', IF_GETOPT_TYPE_BOOLEAN, "debug", "", 0, "Opera como \"man in the middle\" para debug."},
	{0, 0, 0, 0, 0, 0, 0}
};


// ///////////////////////////////////////////////////////////////////// //
void ZK_show_packet(struct sockaddr_in *remote_addr, unsigned char *pack, int len)
{
	ZKHEADER header;
	char buf[30];
	char ip[20];
	int port, i;

	snprintf(ip, sizeof(ip), "%s", inet_ntoa(remote_addr->sin_addr));
	port = ntohs(remote_addr->sin_port);

	fprintf(stdout, "%s\n", siin_getTime(time(NULL), buf, sizeof(buf)));
	fprintf(stdout, "Origem: %s:%d - %d bytes\n", ip, port, len);

	if (len < sizeof(ZKHEADER))
	{
		fprintf(stderr, "Erro: Pacote invalido.\n");
		return;
	}

	memcpy(&header, pack, sizeof(header));

	fprintf(stdout, "CMD: %s\n", ZK_getCMD(header.cmd));
	fprintf(stdout, "cmd: %04X  check: %04X  Session: %04X  Reply: %04X\n", 
		header.cmd, header.checksum, header.sessionid, header.replyid);
	for (i = sizeof(ZKHEADER) ; i < len ; i++)
		if ((pack[i] < ' ') || (pack[i] > 126))
			fprintf(stdout, "%02X ", pack[i]);
		else
			fprintf(stdout, "%c(%02X) ", pack[i], pack[i]);

	fprintf(stdout, "\n\n");
	fflush(stdout);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int ZK_delete_all(IFDEVICE *dev, JSON_VALUE *users)
{
	unsigned char data[MAXPACKET];
	ZKHEADER header;
	int n;
	int r = 0;

	ZK_open(dev, 0);
	n = ZK_sendrecv(dev, CMD_DISABLEDEVICE, data, 0, &header);
	n = ZK_sendrecv(dev, CMD_CLEAR_DATA, data, 0, &header);
	if (header.cmd == CMD_ACK_OK)
		verboseINFO(&(dev->log), "Reset realizado.\n"); 
	else
		r = 1;
	n = ZK_sendrecv(dev, CMD_REFRESHDATA, data, 0, &header);
	n = ZK_sendrecv(dev, CMD_ENABLEDEVICE, data, 0, &header);
	ZK_close(dev);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int ZK_run_debug(char *ip, char *port)
{
	int sock_orig, sock_dest, n, port_orig, test;
	struct sockaddr_in remote_addr;
	unsigned char buf[MAXPACKET];
	char ip_orig[20];
	uint16_t cmd;

	sock_orig = ZK_openUDP(atoi(port));
	if (sock_orig < 1)
		return(sock_orig);

	do
	{
		n = ZK_recv_datagram(sock_orig, buf, MAXPACKET, &remote_addr);
		if (n == 0)
		{
			if_sleep(1000);
			continue;
		}

		ZK_show_packet(&remote_addr, buf, n);
		memcpy(&cmd, buf, sizeof(cmd));
	} while ((cmd == CMD_CHECK) || (n < 1));

	if (n < 0)
	{
		fprintf(stderr, "Falha ao tentar ler pacote.\n");
		return(-1);
	}

	snprintf(ip_orig, sizeof(ip_orig), "%s", inet_ntoa(remote_addr.sin_addr));
	port_orig = ntohs(remote_addr.sin_port);

	sock_dest = ZK_openUDP(port_orig);
	if (sock_dest < 1)
		return(sock_dest);

	for (test = 0 ; ; test = 0)
	{
		if (n > 0)
		{
			ZK_show_packet(&remote_addr, buf, n);
			ZK_send_datagram(sock_dest, ip, atoi(port), buf, n);
			test++;
		}

		n = ZK_recv_datagram(sock_dest, buf, MAXPACKET, &remote_addr);

		if (n > 0)
		{
			ZK_show_packet(&remote_addr, buf, n);
			ZK_send_datagram(sock_orig, ip_orig, port_orig, buf, n);
			test++;
		}

		n = ZK_recv_datagram(sock_orig, buf, MAXPACKET, &remote_addr);

		if (test == 0)
			if_sleep(50);
	}

	close(sock_orig);
	close(sock_dest);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int ZK_run(char *host, char *port)
{
	char *fields[] = {"nro","1", "codigo","0", "verbosity","5", "modelo","ZK", "ip",host, "porta",port, NULL,NULL};
	JSON_VALUE *jconfig = json_object_new_list(fields);
	JSON_VALUE *users, *res, *offs;
	struct tm *dt;
	IFDEVICE dev;
	FILE *fd = NULL;
	int r = 0;

	IFPONTO_device_init(&dev, jconfig);
	ZK_init(&dev);

	time_t ut = time(NULL);
	ut = ZK_getTime4J(&dev, 0);
	dt = localtime(&ut);

	verboseINFO(&(dev.log), "%02d/%02d/%d %02d:%02d:%02d (%X)\n",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min, dt->tm_sec, ut);

	ZK_setTime4J(&dev, 0);

	JSON_VALUE *info = ZK_getInfo4J(&dev, 0);
	verboseINFO(&(dev.log), "%s\n", json_serialize(info));

	if (if_getopt_isChecked(configs, "log"))
	{
		offs = ZK_getEvents4J(&dev, 0);
		json_serialize_file(offs, "ZKlogs.json");
	}

	if (if_getopt_isChecked(configs, "read"))
	{
		users = ZK_getUsers4J(&dev, 0);
		json_serialize_file(users, "ZKusers.json");
	}

	if (if_getopt_isChecked(configs, "delete-all"))
	{
		users = ZK_getUsers4J(&dev, 0);
		r = ZK_delete_all(&dev, users);
	}

	char *jfile = if_getopt_getValue(configs, "update");
	if (jfile[0] != 0)
	{
		fd = fopen(jfile, "r");
		if (fd != NULL)
		{
			users = json_parse_file(fd);
			fclose(fd);

			res = ZK_sendUsers4J(&dev, users);
			verboseINFO(&(dev.log), "%s\n", json_serialize(res));
		}
		else
			verboseERROR(&(dev.log), "Falha ao tentar ler: '%s'\n", jfile);
	}
	
	return(r);
	// TODO
	//ZK_online(&dev, NULL, NULL);
}
// ///////////////////////////////////////////////////////////////////// //


#define DESCRIPT	"Modulo 'standalone' para configuracao/operacao de equipamentos " \
			"de controle de acesso ZKTeco."


// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char **argv)
{
	int r = if_getopt(configs, argc, argv);

	if ((r < 0) || (!if_getopt_isChecked(configs, "host")))
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -h <IP> [df]\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\tshell$ %s -h 192.168.1.201 -d\n", argv[0]);
		fprintf(stderr, "\n");

		fprintf(stderr, "\nExemplo de arquivo JSON:\n\n");
		fprintf(stderr, "[{\"codigo\":\"589\",\"cracha\":\"524877\",\"nome\":\"Nome com 8 caracteres\",\"privilege\":\"0|8\"},{...},{...}]\n\n");
		return(1);
	}

	char *host = if_getopt_getValue(configs, "host");
	char *port = if_getopt_getValue(configs, "port");
	r = 0;

	if (if_getopt_isChecked(configs, "debug"))
		ZK_run_debug(host, port);
	else
		r = ZK_run(host, port);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

#endif
