#include <ifractal.h>

#define MODULE		"FILESYSTEM"


// //////////////////////////////////////////////////////////////////////
FILE_CONTEXT * fs_file_new(char *path, char *name)
{
#ifdef WIN32
	struct _stat filestat;
#else
	struct stat filestat;
	int status;
#endif
	FILE_CONTEXT *ctx;

	ctx = if_malloc(sizeof(FILE_CONTEXT));

	ctx->path = if_strdup(path);
	ctx->name = if_strdup(name);

	ctx->filename = if_malloc(BUFFER_LEN);
	snprintf(ctx->filename, BUFFER_LEN, "%s%c%s", path, PATH_SEPARATOR, name);

#ifdef WIN32
	_stat(ctx->filename, &filestat);
#else
	status = lstat(ctx->filename, &filestat);
	if(status == -1)
		verbose(stderr, "Falha ao tentar ler propriedades de '%s'\n", ctx->filename);
#endif

	ctx->ctime = filestat.st_ctime;
	ctx->size = filestat.st_size;

	return(ctx);
}
// //////////////////////////////////////////////////////////////////////
int fs_file_free(FILE_CONTEXT *ctx)
{
	if (ctx == NULL)
		return(-1);

	if_free(ctx->path);
	if_free(ctx->name);
	if_free(ctx->filename);
	if_free(ctx);

	return(0);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int fs_get_content(_IN FILE *fd, _OUT char **ref)
{
	int size = BUFFER_LEN;
	int len, n;
	char *buf = if_malloc(size);
	char *p;

	if ((fd == NULL) || (ref == NULL))
		return(-1);

	for (p = buf, len = p - buf ; (n = fread(p, 1, size - len, fd)) > 0 ; )
	{
		p += n;
		len = p - buf;
		if (len > (size / 2))
		{
			size = 1 + (size * 2);
			buf = realloc(buf, size);
			p = buf + len;
		}
		*p = 0;
	}

	*ref = buf;
	return(len);
}
// //////////////////////////////////////////////////////////////////////
int fs_file_get_content(FILE_CONTEXT *ctx, _OUT uint8_t **content)
{
	FILE *fd;
	int len;

	fd = fopen(ctx->filename, "rb");
	if (fd == NULL)
		return(-1);

	len = fs_get_content(fd, (char **) content);

	fclose(fd);

	return(len);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_file_check(FILE_CONTEXT *fctx, _IN char *prefix, _IN char *suffix)
{
	int prefix_len = strlen(prefix);
	int suffix_len = strlen(suffix);
	int len = strlen(fctx->name);

	// Verifica se deve usar Expressao Regular
	if (suffix_len == 0)
	{
		if (re_match_direct(prefix, (const char *) fctx->name) < 0)
			return(0);
		else
			return(1);
	}

	if (len < (prefix_len + suffix_len))
		return(0);

	if (strncmp(fctx->name, prefix, prefix_len) != 0)
		return(0);

	if (strcmp(fctx->name + len - suffix_len, suffix) != 0)
		return(0);

	return(1);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PUBLIC DIR_CONTEXT * fs_dir_new(char *path)
{
	DIR_CONTEXT *ctx;
	DIR *dir;

	if (path == NULL)
		return(NULL);

	if ((dir = opendir(path)) == NULL)
	{
		fprintf(stderr, "|%s| Falha ao tentar abrir diretorio: '%s'.\n", MODULE, path);
		return(NULL);
	}

	ctx = if_malloc(sizeof(DIR_CONTEXT));

	ctx->dir = dir;
	strncpy(ctx->path, path, BUFFER_LEN);

	return(ctx);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_dir_free(DIR_CONTEXT *ctx)
{
	if (ctx == NULL)
		return(-1);

	closedir(ctx->dir);
	if_free(ctx);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_dir_iter(DIR_CONTEXT *ctx, FS_CALLBACK cb, void *user_data)
{
	struct dirent *item;
	FILE_CONTEXT *fctx;
	int q, r;

	if ((ctx == NULL) || (cb == NULL))
		return(-1);

	for (q = 0 ; (item = readdir(ctx->dir)) != NULL ; q++, fs_file_free(fctx))
	{
		fctx = fs_file_new(ctx->path, item->d_name);
		if (fctx == NULL)
			continue;

		if ((strcmp(fctx->name, ".") == 0) || (strcmp(fctx->name, "..") == 0))
			continue;

#ifdef WIN32
		struct _stat file;

		fctx->type = FILE_TYPE_REGULAR;

		_stat(fctx->filename, &file);
		if (S_ISDIR(file.st_mode))
			fctx->type = FILE_TYPE_DIR;
#else
		fctx->type = FILE_TYPE_OTHER;

		if (item->d_type == DT_DIR)
			fctx->type = FILE_TYPE_DIR;

		if(item->d_type == DT_REG)
			fctx->type = FILE_TYPE_REGULAR;
#endif

		r = cb(ctx, fctx, user_data);
		if (r)
			break;
	}

	return(q);
}
// //////////////////////////////////////////////////////////////////////



// //////////////////////////////////////////////////////////////////////
FILE_CONTEXT * fs_file_sort_context_new(
	FILE_CONTEXT *first, 
	FILE_CONTEXT *current, 
	FS_SORT_CALLBACK scb, 
	uint8_t descendent, 
	void *user_data)
{
	if (current == NULL)
		return(NULL);

	FILE_CONTEXT *nfc = if_malloc(sizeof(FILE_CONTEXT));
	nfc->name = if_strdup(current->name);
	nfc->path = if_strdup(current->path);
	nfc->filename = if_strdup(current->filename);
	nfc->ctime = current->ctime;
	nfc->size = current->size;
	nfc->type = current->type;
	nfc->next = first;

	FILE_CONTEXT *head = nfc;
	FILE_CONTEXT *a, *b, *ptr, *last;
	int cmp;

	for (ptr = nfc, last = NULL ; (ptr != NULL) && (ptr->next != NULL); ptr = last->next)
	{
		a = ptr;
		b = ptr->next;

		cmp = scb(a, b, user_data);

		if ((cmp == 0) || ((descendent) && (cmp > 0)) || ((!descendent) && (cmp < 0)))
		{
			last = a;
			continue;
		}

		if (a == head)
			head = b;
		else
			last->next = b;

		a->next = b->next;
		b->next = a;

		last = b;
	}

	return(head);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int fs_dir_sorted_iter_callback(DIR_CONTEXT *ctx, FILE_CONTEXT *fctx, void *data)
{
	void **pars = (void **) data;
	FILE_CONTEXT **first = (FILE_CONTEXT **) pars[0];
	FS_SORT_CALLBACK sortcb = (FS_SORT_CALLBACK) pars[1];
	uint8_t *desc = (uint8_t *) pars[2];
	void *user_data = pars[3];

	*first = fs_file_sort_context_new(*first, fctx, sortcb, *desc, user_data);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
int fs_dir_sorted_iter(DIR_CONTEXT *ctx, FS_CALLBACK filecb, FS_SORT_CALLBACK sortcb, uint8_t descendent, void *user_data)
{
	FILE_CONTEXT *gc, *ptr = NULL;
	void *pars[] = {&ptr, sortcb, &descendent, user_data};
	int qty = fs_dir_iter(ctx, fs_dir_sorted_iter_callback, pars);
	int r;

	while (ptr != NULL)
	{
		r = filecb(ctx, ptr, user_data);
		gc = ptr;
		ptr = ptr->next;
		fs_file_free(gc);

		if (r)
			break;
	}

	return(qty);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int fs_dir_name_sort_callback(FILE_CONTEXT *a, FILE_CONTEXT *b, void *user_data)
{
	return(strcmp(a->name, b->name));
}
// //////////////////////////////////////////////////////////////////////
int fs_dir_name_sorted_iter(DIR_CONTEXT *ctx, FS_CALLBACK cb, uint8_t descendent, void *user_data)
{
	int r = fs_dir_sorted_iter(ctx, cb, fs_dir_name_sort_callback, descendent, user_data);
	return(r);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int fs_dir_time_sort_callback(FILE_CONTEXT *a, FILE_CONTEXT *b, void *user_data)
{
	return(b->ctime - a->ctime);
}
// //////////////////////////////////////////////////////////////////////
int fs_dir_time_sorted_iter(DIR_CONTEXT *ctx, FS_CALLBACK cb, uint8_t descendent, void *user_data)
{
	int r = fs_dir_sorted_iter(ctx, cb, fs_dir_time_sort_callback, descendent, user_data);
	return(r);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int fs_dir_size_sort_callback(FILE_CONTEXT *a, FILE_CONTEXT *b, void *user_data)
{
	return(a->size - b->size);
}
// //////////////////////////////////////////////////////////////////////
int fs_dir_size_sorted_iter(DIR_CONTEXT *ctx, FS_CALLBACK cb, uint8_t descendent, void *user_data)
{
	int r = fs_dir_sorted_iter(ctx, cb, fs_dir_size_sort_callback, descendent, user_data);
	return(r);
}
// //////////////////////////////////////////////////////////////////////



// //////////////////////////////////////////////////////////////////////
_CALLBACK int fs_proc_iter_dir_iter(DIR_CONTEXT *dir, FILE_CONTEXT *file, void *data)
{
	void **params = (void **) data;
	PROC_LIST_CALLBACK cb = params[0];
	void *user_data = params[1];
	int pid, ppid, qty_threads;
	char path[PATH_LEN], buf[BUFFER_LEN];
	char *tk[PATH_LEN];
	int fd, n;

	if ((file->name[0] > '9') || (file->name[0] < '0'))
		return(0);

	snprintf(path, PATH_LEN, "%s/stat", file->filename);
	fd = open(path, O_RDONLY);
	n = read(fd, buf, BUFFER_LEN);
	close(fd);
	buf[n] = 0;

	tokenizer(' ', buf, tk, PATH_LEN);
	
	tk[1][strlen(tk[1]) - 1] = 0;
	pid = atoi(tk[0]);
	ppid = atoi(tk[3]);
	qty_threads = atoi(tk[19]);
	cb(tk[1] + 1, pid, ppid, qty_threads, user_data);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_proc_iter(PROC_LIST_CALLBACK cb, void *user_data)
{
#ifdef WIN32
	return(win_proc_iter(cb, user_data));
#else
	DIR_CONTEXT *dir;
	void *params[2];
	int r;

	if (cb == NULL)
		return(-1);

	dir = fs_dir_new("/proc");
	if (dir == NULL)
		return(-2);

	params[0] = cb;
	params[1] = user_data;
	r = fs_dir_iter(dir, fs_proc_iter_dir_iter, params);
	fs_dir_free(dir);

	return(r);
#endif
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int convertMb(int block, int valor)
{
	double ret = (double) block;
	ret *= valor;
	ret /= (1024 * 1024);

	return((int) ret);
}
// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_hd_iter(FS_HD_CALLBACK cb, void *user_data)
{
#ifdef WIN32
	return(0);
#else
	struct statfs fs;
	char *tk[MTAB_LEN + 1];
	char buf[PATH_LEN];
	FILE *fd = fopen("/etc/mtab", "r");
	int uso, n;

	if (fd == NULL)
		return(-1);

	for (n = 0 ; fgets(buf, PATH_LEN, fd) != NULL ; n++)
	{
		if (tokenizer(' ', buf, tk, MTAB_LEN) < MTAB_LEN)
			continue;
			
		if (statfs(tk[MTAB_DIRETORIO], &fs))
			continue;
        
		if (fs.f_blocks == 0)
			continue;
		
		uso = 100 * convertMb(fs.f_bsize, fs.f_blocks - fs.f_bavail) / 
				convertMb(fs.f_bsize, fs.f_blocks);
				
		cb(	tk[MTAB_DEVICE],
			tk[MTAB_DIRETORIO],
			tk[MTAB_TIPO],
			convertMb(fs.f_bsize, fs.f_blocks),	// total
			uso,
			convertMb(fs.f_bsize, fs.f_bavail), 	// livre
			user_data);
	}

	fclose(fd);

	return(n);
#endif
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int fs_delete(char *filename)
{
#ifdef WIN32
	int r = DeleteFile(filename);

	if (r == 0)
		r = GetLastError();
	else
		r = 0;
#else
	int r = unlink(filename);
#endif

	if (r != 0)
	{
		verbose(stderr, "Erro (%d) ao tentar apagar arquivo: '%s'.\n", r, filename);
		return(1);
	}

	return(0);
}
// //////////////////////////////////////////////////////////////////////


#ifdef STANDALONE

int fs_cb(DIR_CONTEXT *ctx, FILE_CONTEXT *fctx, void *user_data)
{
	int *l = (int *) user_data;
	DIR_CONTEXT *rctx;
	int dias, i, k;

	for (i = 0 ; i < *l ; i++)
		fprintf(stdout, "\t");

	dias = (time(NULL) - fctx->ctime) / (24 * 3600);
	fprintf(stdout, "%3d dias  -  %8d %s\n", dias, fctx->size, fctx->name);

	if ((strcmp(fctx->name, ".") == 0) || (strcmp(fctx->name, "..") == 0))
		return(0);

	if (fctx->type == FILE_TYPE_DIR)
	{
		rctx = fs_dir_new(fctx->filename);
		k = *l + 1;
		fs_dir_iter(rctx, fs_cb, &k);
		fs_dir_free(rctx);
	}

	return(0);
}

int main(int argc,char *argv[])
{
	DIR_CONTEXT *ctx;
	int l = 0;

	if (argc < 2)
	{
		fprintf(stderr, "\n\nTeste filesystem.\n\n");
		fprintf(stderr, "\nUso:\n\tshell$ %s <PATH>\n\n", argv[0]);
		return(1);
	}

	ctx = fs_dir_new(argv[1]);
	fs_dir_iter(ctx, fs_cb, &l);
	fs_dir_free(ctx);

	return(0);
}

#endif
