#include <ifractal.h>
#include <ifdevice.h>
#include <idx.h>


// ///////////////////////////////////////////////////////////////////// //
void CID_mirror(void *in, void *out, int bytes)
{
	unsigned char *intaddr = (unsigned char *) in;
	unsigned char *destaddr = (unsigned char *) out;
	int i;

	for (i = 0 ; i < bytes ; i++)
		destaddr[i] = intaddr[bytes - i - 1];
}
// ///////////////////////////////////////////////////////////////////// //
uint16_t CID_Bigendian2int16(IFDEVICE *cid, unsigned char *addr)
{
	uint16_t ret = 0;

	verboseDEBUG(&(cid->log), "CID_Bigendian2int16 - %X\n", addr);

	if (addr != NULL)
		CID_mirror(addr, &ret, 2);

	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //
uint32_t CID_Bigendian2int32(unsigned char *addr)
{
	uint32_t ret = 0;
	CID_mirror(addr, &ret, sizeof(uint32_t));
	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //
void CID_Integer2Bigendian(void *in, void *out, int bytes)
{
	CID_mirror(in, out, bytes);
}
// ///////////////////////////////////////////////////////////////////// //
uint64_t CID_strdec2int64(char *in)
{
	uint64_t ret = 0;
	int i;

	for (i = 0 ; in[i] != 0 ; i++)
	{
		ret *= 10;
		ret += in[i] - '0';
	}

	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
uint64_t CID_getPIS(_IN char *str_pis, unsigned char pis[])
{
	uint64_t r = CID_strdec2int64(str_pis);
	CID_mirror(&r, pis, 8);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void CID_addResult(JSON_VALUE *user, JSON_VALUE *res, char *msg, int err)
{
	JSON_VALUE *reg = json_object_new(1);
	char *cod = json_object_get_string(user, "cod");
	char *codigo = json_object_get_string(user, "codigo");

	json_array_add(res, reg);
	json_object_add(reg, "cod", json_string_new(cod));
	json_object_add(reg, "codigo", json_string_new(codigo));
	json_object_add(reg, "status", json_string_new(msg));
	json_object_add(reg, "msg", json_string_new(msg));
	json_object_add(reg, "cod_error", json_integer_new(err));
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int CID_send(
	_IN IFDEVICE *cid, 
	_IN unsigned char cmd[],
	_IN unsigned char *data, 
	_IN size_t len)
{
	CIDHEADER header;
	char *log;
	int n;

	memcpy(header.cmd, cmd, sizeof(header.cmd));
	CID_Integer2Bigendian(&len, &(header.length), 2);

	n = send_bytes(cid->sock, &header, sizeof(header), 5);
	if (n < sizeof(header))
	{
		verboseFATAL(&(cid->log), "Falha envio (header): %d/%d\n", n, sizeof(header));
		return(-1);
	}
	
	log = siin_hexlog((unsigned char *) &header, sizeof(header));
	verboseDEBUG(&(cid->log), "HEADER iFractal --> leitor   bytes: %d\n%s\n\n", n, log);
	if_free(log);

	if ((data == NULL) || (len < 1))
		return(0);

	n = send_bytes(cid->sock, data, len, 5);
	if (n < len)
	{
		verboseFATAL(&(cid->log), "(%d) Falha envio: %d/%d\n", cid->nro, n, len);
		return(-2);
	}
	
	log = siin_hexlog((unsigned char *) data, len);
	verboseDEBUG(&(cid->log), "DATA iFractal --> leitor   bytes: %d\n%s\n\n", n, log);
	if_free(log);
	fflush(stdout);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_recv(
	_IN IFDEVICE *cid, 
	_OUT unsigned char **data, 
	_INOUT CIDHEADER *header)
{
	unsigned char *buf;
	int len, n;
	char *log;

	n = read_bytes(cid->sock, header, 4, 5);
	if (n < 4)
	{
		verboseFATAL(&(cid->log), 
			"(%d) Falha recv1: %d/%d\n", cid->nro, n, 4);
		return(-1);
	}
	
	log = siin_hexlog((unsigned char *) header, 4);
	verboseDEBUG(&(cid->log), "HEADER iFractal <-- leitor   bytes: %d\n%s\n\n", n, log);
	if_free(log);

	len = CID_Bigendian2int16(cid, header->length);
	if (len <= 0)
		return(len);

	buf = if_malloc(len + 1);
	n = read_bytes(cid->sock, buf, len, 40);

	if (n > 0)
	{	
		log = siin_hexlog((unsigned char *) buf, n);
		verboseDEBUG(&(cid->log), "DATA iFractal <-- leitor   bytes: %d\n%s\n\n", n, log);
		if_free(log);
	}

	if (n < len)
	{
		if_free(buf);
		verboseFATAL(&(cid->log), "Falha recv2: %d/%d\n", n, len);
		return(-2);
	}

	if (data == NULL)
		if_free(buf);
	else
		*data = buf;

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int CID_open(IFDEVICE *cid)
{
	unsigned char cmd[] = {CMD_CONN, CMD_CONN_START};
	unsigned char data[] = {0xF1, 0x0D, 0xA1, 0x5E, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00};
	CIDHEADER header;
	int n, r = -10;

	cid->sock = openTCP(cid->host, cid->port, 5);
	if (cid->sock < 1)
		return(cid->sock);

	n = CID_send(cid, cmd, data, sizeof(data));
	if (n < 1)
	{
		r = -3;
		goto CID_open_err;
	}

	n = CID_recv(cid, NULL, &header);
	if (n < 1)
	{
		r = -4;
		goto CID_open_err;
	}

	return(cid->sock);

CID_open_err:
	if_closesocket(cid->sock);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int CID_close(IFDEVICE *dev)
{
	unsigned char cmd[] = {CMD_CONN, CMD_CONN_END};
	int n;

	n = CID_send(dev, cmd, NULL, 0);
	if_closesocket(dev->sock);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
time_t CID_getTime(IFDEVICE *dev)
{
	unsigned char cmd[] = {CMD_INFO, CMD_INFO_DATETIME};
	unsigned char *data = NULL;
	CIDHEADER header;
	struct tm dt;
	int n;

	n = CID_send(dev, cmd, NULL, 0);
	n = CID_recv(dev, &data, &header);
	if (n < 1)
		return(n);

	dt.tm_sec = data[0];
	dt.tm_min = data[1];
	dt.tm_hour = data[2];
	dt.tm_mday = data[3];
	dt.tm_mon = data[4] - 1;
	dt.tm_year = data[5] +  100;

	if_free(data);

	return(mktime(&dt));
}
// ///////////////////////////////////////////////////////////////////// //
int CID_setTime(IFDEVICE *cid, int timediff)
{
	unsigned char cmd[] = {CMD_INFO, CMD_INFO_SET_DATETIME};
	time_t now = time(NULL);
	unsigned char data[6];
	CIDHEADER header;
	struct tm *dt;
	int n;

	now += 3600 * timediff;
	dt = localtime(&now);

	data[0] = dt->tm_sec;
	data[1] = dt->tm_min;
	data[2] = dt->tm_hour; 
	data[3] = dt->tm_mday;
	data[4] = dt->tm_mon + 1;
	data[5] = dt->tm_year - 100;

	n = CID_send(cid, cmd, data, sizeof(data));
	n = CID_recv(cid, NULL, &header);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PUBLIC int CID_getInfo(IFDEVICE *cid, _INOUT CIDINFO **info_out)
{
	unsigned char cmd[] = {CMD_INFO, CMD_INFO_STATUS};
	CIDINFO *info = NULL;
	CIDHEADER header;
	int n;

	n = CID_send(cid, cmd, NULL, 0);
	n = CID_recv(cid, (unsigned char **) &info, &header);
	if (n == sizeof(CIDINFO))
		*info_out = info;
	else
		*info_out = NULL;

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
_PRIVATE JSON_VALUE * CID_getAFDMarcacao(char *line, JSON_VALUE *obj)
{
	char buf[PATH_LEN];

	snprintf(buf, PATH_LEN, "20%c%c-%c%c-%c%c %c%c:%c%c:00", 
		line[16], line[17], 	// ano
		line[12], line[13], 	// mes
		line[10], line[11], 	// dia
		line[18], line[19], 	// hora
		line[20], line[21] 	// min
		);

	json_object_add(obj, "datahora", json_string_new(buf));

	memset(buf, 0, sizeof(buf));
	memcpy(buf, line + 22, 12);
	if (buf[11] == ' ')
		buf[11] = 0;
	json_object_add(obj, "pis", json_string_new(buf));

	return(obj);
}
// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int CID_coleta_perform(IFDEVICE *dev, char *afd, JSON_VALUE *offs)
{
	JSON_VALUE *obj;
	char *pi, *p = afd, tipo_registro;
	int current_nsr;

	do
	{
		obj = json_object_new(2);

		tipo_registro = *(p + 9);
		switch (tipo_registro)
		{
			case '3':
				CID_getAFDMarcacao(p, obj);
				json_array_add(offs, obj);
				break;

			case '1':
			case '2':
			case '4':
			case '5':
				json_array_add(offs, obj);
				break;

			default:
				verboseINFO(&(dev->log), "AFD tipo invalido: %02X\n", tipo_registro);
				break;
		}

		*(p + 9) = 0;
		if (atoi(p) > 0)
			current_nsr = atoi(p);
		*(p + 9) = tipo_registro;

		// TODO - eliminar caracteres especiais
		for (pi = p ; *p != 0 ; p++)
		{
			if (*p == '\r')
				*p = 0;

			if (*p == '\n')
			{
				*p++ = 0;
				break;
			}
		}

		json_object_add(obj, "afd", json_string_new(pi));
		json_object_add(obj, "nsr", json_integer_new(current_nsr));
	}
	while (*p != 0);

	if (current_nsr < 0)
		return(1);

	((CID *) dev)->last_nsr = current_nsr;

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_coleta_in(IFDEVICE *dev, CIDAFDADDRESS *afd_address, JSON_VALUE *offs)
{
	unsigned char cmd[] = {CMD_AFD, CMD_AFD_COLETA};
	unsigned char *buf = NULL, data[8];
	int posicao = CID_Bigendian2int16(dev, afd_address->posicao);
	CIDHEADER header;
	int n, qtd_blocos;

	qtd_blocos = CID_Bigendian2int32(afd_address->total_blocos);
	qtd_blocos -= CID_Bigendian2int16(dev, afd_address->bloco);
	
	memset(data, 0, sizeof(data));
	memcpy(data + 2, afd_address->bloco, 2);

	data[4] = CID_QTD_BLOCOS;
	if (qtd_blocos < CID_QTD_BLOCOS)
		data[4] = (unsigned char) qtd_blocos + 1;

	CID_send(dev, cmd, data, sizeof(data));
	n = CID_recv(dev, &buf, &header);
	if (n < 0)
	{
		verboseWARN(&(dev->log), "Falha ao tentar realizar coleta.\n");
		return(-4);
	}
		
	if (n >= 34)
		CID_coleta_perform(dev, (char *) buf + posicao, offs);

	if_free(buf);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_coleta(IFDEVICE *dev, uint32_t nsr, JSON_VALUE *offs)
{
	unsigned char cmd[] = {CMD_AFD, CMD_AFD_NSR};
	unsigned char data[] = {0,0,0,0};
	CIDAFDADDRESS *afd_address = NULL;
	CIDHEADER header;
	int n;

	CID_Integer2Bigendian(&nsr, data, 4);
	n = CID_send(dev, cmd, data, sizeof(data));
	if (n < 0)
	{
		verboseERROR(&(dev->log), "Falha ao tentar iniciar coleta. (comunicacao)\n");
		return(-2);
	}

	n = CID_recv(dev, (unsigned char **) &afd_address, &header);
	if ((n < sizeof(CIDAFDADDRESS)) || (afd_address == NULL))
	{
		verboseWARN(&(dev->log), "Falha ao tentar iniciar coleta. (NSR aparentemente travado)\n");
		((CID *) dev)->last_nsr = -1;
		return(-3);
	}

	verboseINFO(&(dev->log), "Coleta nsr: %d.\n", nsr);
	CID_coleta_in(dev, afd_address, offs);
	if_free(afd_address);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int CID_excluiUser(IFDEVICE *dev, CIDUSER *user)
{
	unsigned char cmd[] = {CMD_AFD, CMD_DEL_PIS};
	CIDHEADER header;
	int n;

	CID_send(dev, cmd, (unsigned char *) user->pis, sizeof(user->pis));
	n = CID_recv(dev, NULL, &header);
	if (n < 0)
	{
		verboseWARN(&(dev->log), "Falha ao tentar excluir funcionario.\n");
		return(-1);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_sendUser(IFDEVICE *dev, CIDUSER *user)
{
	unsigned char cmd[] = {CMD_AFD, CMD_SET_USER};
	CIDHEADER header;
	int n;

	CID_send(dev, cmd, (unsigned char *) user, sizeof(CIDUSER));
	n = CID_recv(dev, NULL, &header);
	if (n < 0)
		return(-1);

	if (header.cmd[0] == CMD_SET_USER_ERR)
	{
		verboseWARN(&(dev->log), "Falha ao tentar incluir funcionario.\n");
		return(-2);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int is_admin_iter(JSON_VALUE *jadmin, void *user_data)
{
	void **ctx = (void **) user_data;
	char *codigo = ctx[0];
	char **senha = ctx[1];
	char *admin_cod = json_object_get_string(jadmin, "codigo");

	if(strcmp(admin_cod, codigo) != 0)
		return(0);
	
	*senha = json_object_get_string(jadmin, "senha");

	return(1);
}
// ///////////////////////////////////////////////////////////////////// //
char * is_admin(IFDEVICE *dev, char *codigo)
{
	JSON_VALUE *admins = json_object_find(dev->config, "admins");
	char *senha = "";

	if((admins == NULL) || (json_get_type(admins) != JSON_ARRAY_TYPE))
		return("");
	
	void *ctx[] = {codigo, &senha};
	json_array_iter(admins, is_admin_iter, ctx);

	return(senha);
}
// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int CID_load_list_iter(JSON_VALUE *juser, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	JSON_VALUE *res = (JSON_VALUE *) params[1];
	char msg[PATH_LEN];
	int l, value, r;
	CIDUSER user;
	char *strpis = json_object_get_string(juser, "pis");
	char *codigo = json_object_get_string(juser, "codigo");
	char *nome = json_object_get_string(juser, "nome");
	char *senha = is_admin(dev, codigo);
	char *cracha = json_object_get_string(juser, "cracha");
	char *oper = json_object_get_string(juser, "tipo_operacao");

	memset(&user, 0, sizeof(CIDUSER));

	if (senha[0] != 0)
		user.privilegio = 1;

	value = atoi(codigo);
	CID_mirror(&value, &(user.codigo), sizeof(user.codigo));

	snprintf(user.nome, sizeof(user.nome), "%s", nome);
	snprintf(user.senha, sizeof(user.senha), "%s", senha);
	snprintf(user.barras, sizeof(user.barras), "%s", cracha);

	if ((l = strlen(cracha)) > 5)
	{
		value = atoi(cracha + (l - 5));
		cracha[l - 5] = 0;
		value += (atoi(cracha) << 16);
	}
	else
		value = atoi(cracha);

	CID_mirror(&value, &(user.rfid), sizeof(user.rfid));
	CID_getPIS(strpis, user.pis);

	if (oper[0] == '3')
	{
		r = CID_excluiUser(dev, &user);
		if (r == 0)
			snprintf(msg, sizeof(msg), "pis: %s - excluido.", strpis);
	}
	else
	{
		r = CID_sendUser(dev, &user);
		if (r == 0)
			snprintf(msg, sizeof(msg), "pis: %s - incluido.", strpis);
	}

	if (r != 0)
		snprintf(msg, sizeof(msg), "pis: %s - operacao falhou.", strpis);


	CID_addResult(juser, res, msg, r);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int getUser_callback(CIDUSER *user, void *user_data)
{
	JSON_VALUE *jusers = (JSON_VALUE *) user_data;
	JSON_VALUE *juser = json_object_new(1);
	uint32_t rfid, codigo;
	uint64_t pis;
	char buf[16];
	int i;

	user->nome[51] = 0;
	for (i = 50 ; (i > 0) && (user->nome[i] == ' ') ; i--)
		user->nome[i] = 0;

	user->barras[13] = 0;
	for (i = 12 ; (i > 0) && (user->barras[i] == ' ') ; i--)
		user->barras[i] = 0;

	rfid = user->rfid[1] * 100000;
	rfid += user->rfid[2] * 0x100;
	rfid += user->rfid[3];

	for (pis = user->pis[0], i = 1 ; i < 8 ; i++)
	{
		pis *= 256;
		pis += user->pis[i];
	}

	snprintf(buf, sizeof(buf), "%" PRId64, pis);

	CID_mirror(user->codigo, &codigo, 4);

	json_object_add(juser, "nome", json_string_new(user->nome));
	json_object_add(juser, "pis", json_string_new(buf));
	json_object_add(juser, "barras", json_string_new(user->barras));
	json_object_add(juser, "rfid", json_integer_new(rfid));
	json_object_add(juser, "cracha", json_integer_new(rfid));
	json_object_add(juser, "bio_qty", json_integer_new(user->qty_bio));
	json_object_add(juser, "privilegio", json_integer_new(user->privilegio));
	json_object_add(juser, "codigo", json_integer_new(codigo));

	json_array_add(jusers, juser);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_getUsers(IFDEVICE *dev, CID_USER_CALLBACK cb, void *user_data)
{
	unsigned char cmd[] = {CMD_AFD, CMD_GET_USERS};
	int qty_users = 0, n, i, q;
	unsigned char *data = NULL;
	CIDHEADER header;
	CIDUSER *user;

	do
	{
		n = CID_send(dev, cmd, NULL, 0);
		if (n < 0)
			return(-1);

		n = CID_recv(dev, &data, &header);
		if (n < 0)
			return(-2);

		q = n / sizeof(CIDUSER);
		qty_users += q;

		for (i = 0, user = (CIDUSER *) data ; i < q ; i++) 
			cb(user + i, user_data);

		if_free(data);

		if_sleep(2000);

	} while (q >= 100);

	return(qty_users);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_getBio(IFDEVICE *dev, unsigned char *pis, CID_BIO_CALLBACK cb, void *user_data)
{
	unsigned char cmd[] = {CMD_AFD, CMD_GET_USER_BIO};
	unsigned char *data = NULL;
	CIDHEADER header;
	int n, i;

	n = CID_send(dev, cmd, pis, 8);
	if (n < 0)
		return(-1);

	for (i = 0 ; ; i++)
	{
		n = CID_recv(dev, &data, &header);
		if (n < 0)
			return(-2);

		if (n == 0)
			break;

		cb(pis, data, n, user_data);

		if_free(data);
	}

	return(i);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_deleteBio(IFDEVICE *dev, char *pis)
{
	unsigned char cmd[] = {CMD_AFD, CMD_DEL_USER_BIO};
	unsigned char buf[8];
	unsigned char *data = NULL;
	CIDHEADER header;
	int n;

	CID_getPIS(pis, buf);

	n = CID_send(dev, cmd, buf, 8);
	if (n < 0)
		return(-1);

	n = CID_recv(dev, &data, &header);
	if (n < 0)
		return(-2);

	if (n > 0)
		if_free(data);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_sendBio_iter(JSON_VALUE *jtemplate, void *user_data)
{
	void **ctx = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) ctx[0];
	int *q = (int *) ctx[1];
	unsigned char cmd[] = {CMD_AFD, CMD_SEND_USER_BIO};
	unsigned char *data, template[CID_BIO_LEN];
	CIDHEADER header;
	char *strtemp = json_object_get_string(jtemplate, "template");
	char *vendor = json_object_get_string(jtemplate, "vendor");
	int n;

	if (strcmp(vendor, "idclass") != 0)
		return(0);

	n = strlen(strtemp);
	data = hex2bin(strtemp);
	memcpy(template, data, n / 2);
	if_free(data);

	n = CID_send(dev, cmd, template, CID_BIO_LEN);
	if (n < 0)
	{
		*q = -2;
		return(-2);
	}

	n = CID_recv(dev, &data, &header);
	if (n < 0)
	{
		*q = -3;
		return(-3);
	}

	if (n > 0)
		if_free(data);

	*q += 1;

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_sendBio(IFDEVICE *dev, JSON_VALUE *juser)
{
	JSON_VALUE *jtemplates;
	char *str_pis;
	int n = 0;

	jtemplates = json_object_find(juser, "templates");
	str_pis = json_object_get_string(juser, "pis");
	n = CID_deleteBio(dev, str_pis);
	if (!n)
		verboseINFO(&(dev->log), "Exclui biometria PIS: %s.\n", str_pis);

	void *ctx[] = {dev, &n};
	json_array_iter(jtemplates, CID_sendBio_iter, ctx);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int CID_insertBio_iter(unsigned char *pis, unsigned char *data, size_t len, void *user_data)
{
	void **params = (void **) user_data;
	JSON_VALUE *user = (JSON_VALUE *) params[1];
	JSON_VALUE *templates = json_object_find(user, "templates");
	JSON_VALUE *temp = json_object_new(1);
	char *hextemp;

	if (templates == NULL)
	{
		templates = json_array_new(1);
		json_object_add(user, "templates", templates);
	}

	json_array_add(templates, temp);
	json_object_add(temp, "vendor", json_string_new("idclass"));

	hextemp = bin2hex(data, len);
	json_object_add(temp, "template", json_string_new(hextemp));
	if_free(hextemp);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_getBio4j_iter(JSON_VALUE *user, void *user_data)
{
	IFDEVICE *dev = (IFDEVICE *) user_data;
	void *params[] = {dev, user};
	char *strpis = json_object_get_string(user, "pis");
	unsigned char pis[8];

	CID_getPIS(strpis, pis);
	CID_getBio(dev, pis, CID_insertBio_iter, params);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int CID_sendBio_checkVendor(JSON_VALUE *jtemp, void *user_data)
{
	int *n = (int *) user_data;
	char *vendor = json_object_get_string(jtemp, "vendor");

	if (strcmp(vendor, "idclass") == 0)
		*n += 1;
		
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int CID_sendBio4J_iter(JSON_VALUE *user, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	JSON_VALUE *res = (JSON_VALUE *) params[1];
	char *str_pis = json_object_get_string(user, "pis");
	char msg[PATH_LEN];
	int r;

	JSON_VALUE *jtemplates = json_object_find(user, "templates");
	if ((jtemplates == NULL) || (json_array_length(jtemplates) < 1))
	{
		verboseINFO(&(dev->log), "pis: %s - Sem biometria.", str_pis);
		return(0);
	}

	json_array_iter(jtemplates, CID_sendBio_checkVendor, &r);
	if (r < 1)
	{
		CID_addResult(user, res, "Nenhuma biometria compativel.", 0);
		return(0);
	}

	r = CID_sendBio(dev, user);
	if (r > 0)
	{
		snprintf(msg, sizeof(msg), "pis: %s - Biometria enviada.", str_pis);
		verboseINFO(&(dev->log), "%s\n", msg);
	}
	else if (r < 0)
	{
		snprintf(msg, sizeof(msg), "pis: %s - Falha ao tentar enviar biometria.", str_pis);
		verboseERROR(&(dev->log), "%s\n", msg);
	}

	CID_addResult(user, res, msg, r);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// Implementacao //////////////////////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_getInfo4J(IFDEVICE4J *dev4j, long sock)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	CIDINFO *info = NULL;
	JSON_VALUE *jinfo;
	char msg[PATH_LEN];
	int nsr, bobina, tamanho_bobina;
	int r = CID_open(dev);

	if (r < 1)
		return(NULL);
	
	jinfo = json_object_new(1);

	CID_getInfo(dev, &info);
	if (info != NULL)
	{
		nsr = CID_Bigendian2int32(info->nsr);
        
		bobina = 100 * CID_Bigendian2int32(info->restante_bobina);
		tamanho_bobina = CID_Bigendian2int32(info->tamanho_bobina);
		if (tamanho_bobina > 0)
			bobina /= tamanho_bobina;
		else
			bobina = 0;

		if (((CID *) dev)->last_nsr < 0)
			snprintf(msg, PATH_LEN, "nsr: %d (travada) ONLINE Papel: %d%%", nsr, bobina);
		else
			snprintf(msg, PATH_LEN, "nsr: %d - ONLINE Papel: %d%%", nsr, bobina);

		json_object_add(jinfo, "bobina", json_integer_new(bobina));
		json_object_add(jinfo, "msg", json_string_new(msg));
		json_object_add(jinfo, "status", json_string_new(msg));
		json_object_add(jinfo, "serial", json_string_new(info->serial));
		
		if_free(info);
	}

	CID_close(dev);

	return(jinfo);
}
// ///////////////////////////////////////////////////////////////////// //
time_t CID_getTime4J(IFDEVICE4J *dev4j, long sock)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	int r = CID_open(dev);

	if (r < 1)
		return(0);

	time_t dt = CID_getTime(dev);

	CID_close(dev);

	return(dt);
}
// ///////////////////////////////////////////////////////////////////// //
intptr_t CID_setTime4J(IFDEVICE4J *dev4j, intptr_t diff)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	int r = CID_open(dev);

	if (r < 1)
		return(0);

	CID_setTime(dev, diff);

	CID_close(dev);

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_sendUsers4J(IFDEVICE4J *dev4j, JSON_VALUE *users)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	JSON_VALUE *res;
	void *params[2];
	int r = CID_open(dev);

	if (r < 1)
		return(NULL);

	res = json_array_new(1);

	params[0] = dev;
	params[1] = res;

	json_array_iter(users, CID_load_list_iter, params);

	CID_close(dev);
	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_getUsers4J(IFDEVICE4J *dev4j, intptr_t none)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	int r = CID_open(dev);

	if (r < 1)
		return(NULL);

	JSON_VALUE *users = json_array_new(1);

	CID_getUsers(dev, getUser_callback, users);

	CID_close(dev);
	return(users);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_sendBio4J(IFDEVICE4J *dev4j, JSON_VALUE *users)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	JSON_VALUE *res;
	void *params[2];
	int r = CID_open(dev);

	if (r < 1)
		return(NULL);

	res = json_array_new(1);
	params[0] = dev;
	params[1] = res;

	json_array_iter(users, CID_sendBio4J_iter, params);

	CID_close(dev);
	return(res);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_getBio4J(IFDEVICE4J *dev4j, JSON_VALUE *users)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	int r = CID_open(dev);

	if (r < 1)
		return(0);

	json_array_iter(users, CID_getBio4j_iter, dev4j);

	CID_close(dev);
	return(json_clone(users));
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * CID_getEvents4J(IFDEVICE4J *dev4j, intptr_t nsr)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	int r = CID_open(dev);
	JSON_VALUE *offs;

	if (r < 1)
		return(NULL);

	offs = json_array_new(1);
	CID_coleta(dev, nsr, offs);

	CID_close(dev);
	return(offs);
}
// ///////////////////////////////////////////////////////////////////// //
void CID_init(IFDEVICE4J *dev)
{
	dev->getInfo = (IFDEVICE4J_getInfo) CID_getInfo4J;
	dev->getTime = (IFDEVICE4J_getTime) CID_getTime4J;
	dev->setTime = CID_setTime4J;
	dev->sendUsers = CID_sendUsers4J;
	dev->getUsers = CID_getUsers4J;
	dev->sendBio = CID_sendBio4J;
	dev->getBio = CID_getBio4J;
	dev->getEvents = CID_getEvents4J;
	dev->free = NULL;
	((CID *) dev)->last_nsr = 0;
}
// ///////////////////////////////////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //


