/** @file ifractal_utils.h Biblioteca para gerenciamento de threads.
\ingroup 

\note -----
\author Felipe Peressoni Waltrick	felipe@ifractal.com.br
\version 2.0
**/

#ifndef IFRACTAL_THREADS
#define IFRACTAL_THREADS


#include <ifractal.h>


typedef enum
{
	TH_DEAD,
	TH_ZOMBIE,
	TH_ALIVE,
	TH_SUSPENDED,
	TH_LEN
} THREAD_STATE;

typedef enum
{
	IF_EVENT_START,
	IF_EVENT_WAIT,
	IF_EVENT_FREE,
	IF_EVENT_IDLE,
	IF_EVENT_END,
	//IF_EVENT_USER,
	IF_EVENT_LEN
} IF_EVENT_TYPE;

typedef struct _IF_THREAD_LIST IF_THREAD_LIST;
typedef struct _IF_THREAD_EVENT IF_THREAD_EVENT;

// Callbacks
typedef void * (*IF_THREAD_FUNC)(THREAD_STATE *, int id, void *data);
typedef void * (*IF_THREAD_EVENT_CALLBACK)(IF_THREAD_LIST *, IF_THREAD_EVENT *);


/**
Define a "classe" IF_THREAD, tem como objetivo permitir o gerenciamento de uma lista de threads de forma segura através de MUTEX.<BR/>
Alem de compatibilizar uso de threads entre diferentes sistemas.
**/
typedef struct _IF_THREAD
{
	int id;
	THREAD_STATE th_state;
	void *data;
	IF_THREAD_FUNC func;
#ifdef WIN32
	HANDLE thread;
#else
	pthread_t thread;
#endif
} IF_THREAD;


struct _IF_THREAD_EVENT
{
	IF_EVENT_TYPE event;
	int id;
	IF_THREAD *thread;
	void *data;
};


struct _IF_THREAD_LIST
{
	THREAD_STATE th_state;
	IF_THREAD_EVENT_CALLBACK event_callback;
#ifdef WIN32
	HANDLE thread;
	HANDLE mutex;
#else
	pthread_t thread;
	pthread_mutex_t mutex;
#endif
	IF_SLIST *first;
	time_t delay;
};


// Interface publica ////////////////////
IF_THREAD_LIST * if_thread_list_new(IF_THREAD_EVENT_CALLBACK);
IF_THREAD * if_thread_add(IF_THREAD_LIST *, IF_THREAD_FUNC, intptr_t id, void *data);
void if_thread_stop(IF_THREAD_LIST *, intptr_t id);
void if_thread_list_finalize(IF_THREAD_LIST *);
int if_thread_list_wait(IF_THREAD_LIST *);
int if_thread_mutex_lock(IF_THREAD_LIST *);
int if_thread_mutex_unlock(IF_THREAD_LIST *);
IF_THREAD * if_thread_get_id(IF_THREAD_LIST *, intptr_t id);

void * if_thread_event_debug(IF_THREAD_LIST *list, IF_THREAD_EVENT *ev);

typedef int (*THREAD_ITER_FUNC)(IF_THREAD *, void *user_data);
int if_thread_iter(IF_THREAD_LIST *, THREAD_ITER_FUNC, void *user_data);
// //////////////////////////////////////

#endif // IFRACTAL_THREADS_H

