#include <ifractal.h>

#ifdef WIN32
CRITICAL_SECTION CriticalSection;
#else
#include <sys/ioctl.h>
#endif

#ifdef __linux__
#include <linux/rtc.h>
#endif

#include "verbose.c"
#include "getopts.c"


static volatile int mem_count = 0;

#ifdef WIN32
// ////////////////////////////////////////////////////////////////////////
void init_critical()
{
	static int cs = 0;

	if (cs == 0)
	{
		cs++;
		InitializeCriticalSection(&CriticalSection);
		SetCriticalSectionSpinCount(&CriticalSection, 100);
	}
}
// ////////////////////////////////////////////////////////////////////////
#endif


// ////////////////////////////////////////////////////////////////////////
char * if_strdup(char *s)
{
	char *m;

#ifdef WIN32
	init_critical();
#endif
	m = strdup(s);

	mem_count++;
	if (m == NULL)
		verbose(stderr, "mem_count: %d   -   %X = strdup(%s);\n", mem_count, (uintptr_t) m, s);

#ifdef DEBUG
	fprintf(stdout, "mem_count: %d   -   STRDUP(%X);\n", mem_count, (uintptr_t) m);
	fflush(stdout);
#endif

	return(m);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PRIVATE void siin_make_version_string(_INOUT char *version)
{
	struct tm *dt;
	time_t unixtime = RELEASE_UNIXTIME;

	dt = localtime(&unixtime);
	snprintf(version, PATH_LEN, "%d.%d.%d r%d (%02d/%02d/%d %02d:%02d)", 
		FWVER_A,FWVER_B,FWVER_C,FWREV,
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC char * siin_version()
{
	static char *siin_version_str = NULL;

	if (siin_version_str == NULL)
		siin_version_str = if_malloc(PATH_LEN);

	siin_make_version_string(siin_version_str);

	return(siin_version_str);
}
// ////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////// //
_PUBLIC char * siin_get_local_ip(_IN char *iface, _INOUT char *ip, size_t ip_len)
{
#ifdef WIN32
#else
	int fm = AF_INET;
	struct ifaddrs *ifaddr, *ifa;
	int family , s;
	char host[NI_MAXHOST];

	ip[0] = 0;
 
	if (getifaddrs(&ifaddr) == -1) 
	{
		return(NULL);
	}
 
	for (ifa = ifaddr ; ifa != NULL ; ifa = ifa->ifa_next) 
	{
		if (ifa->ifa_addr == NULL)
			continue;
 
		family = ifa->ifa_addr->sa_family;
 
		if(strcmp(ifa->ifa_name, iface) == 0)
		{
			if (family == fm) 
			{
				s = getnameinfo(ifa->ifa_addr, (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6), host, NI_MAXHOST, NULL, 0, NI_NUMERICHOST);
				 
				if (s != 0) 
				{
					fprintf(stderr, "getnameinfo() failed: %s\n", gai_strerror(s));
					return(NULL);
				}
				 
				snprintf(ip, ip_len, "%s", host);
			}
		}
	}
 
	freeifaddrs(ifaddr);
#endif

	return(ip);
}
// ///////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////
_PRIVATE int siin_converte_mac(char *s, unsigned char *mac)
{
	char *p;
	int k, d;

	if (strlen(s) == 12)
		d = 2;
	else
		d = 3;

	// Converte
	for (p = s ; (*p != 0) && (*p != '\n') ; p++)
	{
		k = (p - s) / d;
		if (*p >= 'a')
			mac[k] = 16 * (*p - 'a' + 10);
		else
			mac[k] = 16 * (*p - '0');
		
		p++;
		if (*p >= 'a')
			mac[k] += *p - 'a' + 10;
		else
			mac[k] += *p - '0';

		if (*(p + 1) == ':')
			p++;
	}

	return(0);
}
// ////////////////////////////////////////////////////////////


#ifdef WIN32
// ////////////////////////////////////////////////////////////
_PUBLIC int siin_get_ifaces(SIIN_NET_IFACE cb, void *user_data)
{
	IP_ADAPTER_INFO AdapterInfo[10];
	DWORD dwBufSize = sizeof(AdapterInfo);
	PIP_ADAPTER_INFO pAdapterInfo = AdapterInfo;
	char mac[30];
	DWORD ret;
	int j;

	// Recupera MAC address
	if ((ret = GetAdaptersInfo(AdapterInfo, &dwBufSize)) != NO_ERROR)
	{
		verbose(stderr, "GetAdaptersInfo error: %d\n", ret);
		return(0);
	}

	for (j = 0 ; pAdapterInfo ; j++, pAdapterInfo = pAdapterInfo->Next)
	{
		snprintf(mac, sizeof(mac), "%02x:%02x:%02x:%02x:%02x:%02x",
			(unsigned char) pAdapterInfo->Address[0],
			(unsigned char) pAdapterInfo->Address[1],
			(unsigned char) pAdapterInfo->Address[2],
			(unsigned char) pAdapterInfo->Address[3],
			(unsigned char) pAdapterInfo->Address[4],
			(unsigned char) pAdapterInfo->Address[5]);

		if (strcmp(pAdapterInfo->IpAddressList.IpAddress.String, "0.0.0.0") == 0)
			continue;

		cb(pAdapterInfo->AdapterName, mac, pAdapterInfo->IpAddressList.IpAddress.String, user_data);
	}

	return(j);
}
// ////////////////////////////////////////////////////////////
#else
// ////////////////////////////////////////////////////////////
int fs_cb(DIR_CONTEXT *ctx, FILE_CONTEXT *fctx, void *user_data)
{
	void **params = (void **) user_data;
	SIIN_NET_IFACE cb = (SIIN_NET_IFACE) params[0];
	void *data = (void *) params[1];
	char buf[40], fn[PATH_LEN], ip[40];
	FILE *fd;

	if ((strcmp(fctx->name, ".") == 0) || (strcmp(fctx->name, "..") == 0) || (strcmp(fctx->name, "lo") == 0))
		return(0);

	if (fctx->type != FILE_TYPE_OTHER)
		return(0);

	snprintf(fn, PATH_LEN, "%s/address", fctx->filename);
	fd = fopen(fn, "r");
	if (fd != NULL)
	{
		fgets(buf, PATH_LEN, fd);
		fclose(fd);
	}
	else
		return(1);

	int i;
	for (i = 0 ; buf[i] != 0 ; i++)
		if (buf[i] == '\n')
		{
			buf[i] = 0;
			break;
		}

	if (cb != NULL)
	{
		siin_get_local_ip(fctx->name, ip, sizeof(ip));
		cb(fctx->name, buf, ip, data);
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
_PUBLIC int siin_get_ifaces(SIIN_NET_IFACE cb, void *user_data)
{
	void *params[] = {cb, user_data};
	DIR_CONTEXT *ctx;
	int qty;

	ctx = fs_dir_new("/sys/class/net");
	qty = fs_dir_iter(ctx, fs_cb, params);
	fs_dir_free(ctx);

	return(qty);
}
// ////////////////////////////////////////////////////////////
#endif

// ////////////////////////////////////////////////////////////
_CALLBACK void siin_mac_cb(char *iface, char *macbuf, char *ip, void *user_data)
{
	unsigned char *buf = (unsigned char *) user_data;
	int sum = buf[0];

	int i;
	for (i = 1 ; i < 6 ; i++)
		sum += buf[i];

	if (sum > 0)
		return;
		
	siin_converte_mac(macbuf, buf);
}
// ////////////////////////////////////////////////////////////
_PRIVATE void siin_get_mac(_INOUT unsigned char *mac, _IN int len)
{
	memset(mac, 0, len);

	if (len < MAC_LEN)
		return;

	siin_get_ifaces(siin_mac_cb, mac);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC char * siin_mac(_INOUT char *buf, _IN int buflen)
{
	unsigned char macaddr[MAC_LEN];

	siin_get_mac(macaddr, MAC_LEN);
	snprintf(buf, buflen, "%02X%02X%02X%02X%02X%02X", 
		macaddr[0], macaddr[1], macaddr[2],
		macaddr[3], macaddr[4], macaddr[5]);

	return(buf);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
_PRIVATE void siin_machine_info_proc(_INOUT JSON_VALUE *info)
{
	char *keys[] = {
		"processor", NULL,
		"vendor_id", "vendor",
		"model name", "model",
		NULL, NULL};
	int i, k, j, cpus = 0;
	char buf[PATH_LEN];
	FILE *fd;

	fd = fopen("/proc/cpuinfo", "r");
	if (fd == NULL)
		return;

	while (fgets(buf, PATH_LEN, fd) != NULL)
	{
		for (i = 0 ; keys[i] != NULL ; i += 2)
		{
			if (strncmp(keys[i], buf, strlen(keys[i])) != 0)
				continue;

			if (keys[i + 1] == NULL)
			{
				cpus++;
				continue;
			}

			for (k = 0 ; buf[k] != 0 ; k++)
			{
				if (buf[k] == ':')
				{
					for (j = k + 1 ; buf[j] != 0 ; j++)
						if (buf[j] == '\n')
							buf[j] = 0;

					json_object_add(info, keys[i + 1], json_string_new(buf + k + 2));
					break;
				}
			}
		}
	}

	json_object_add(info, "cores", json_integer_new(cpus));

	fclose(fd);
}
// ////////////////////////////////////////////////////////////
_PRIVATE void siin_machine_info_etc(_INOUT JSON_VALUE *info)
{
	char *files[] = {
		"/etc/issue", "so",
		"/etc/machine-id", "id",
		"/proc/sys/kernel/hostname", "hostname",
		NULL, NULL
	};
	char buf[PATH_LEN];
	int i, k;
	FILE *fd;

	for (k = 0 ; files[k] != NULL ; k += 2)
	{
		fd = fopen(files[k], "r");
		if (fd == NULL)
			continue;

		fgets(buf, PATH_LEN, fd);
		for (i = 0 ; buf[i] != 0 ; i++)
			if ((buf[i] == '\n') || (buf[i] == '\\'))
			{
				buf[i] = 0;
				json_object_add(info, files[k + 1], json_string_new(buf));
				break;
			}

		fclose(fd);
	}
}
// ////////////////////////////////////////////////////////////
_CALLBACK void siin_machine_info_cb(char *iface, char *mac, char *ip, void *user_data)
{
	char *list[] = {"name",iface, "macaddress",mac, "ip",ip, NULL,NULL};
	JSON_VALUE *ifaces = (JSON_VALUE *) user_data;
	JSON_VALUE *jiface = json_object_new_list(list);
	json_array_add(ifaces, jiface);
}
// ////////////////////////////////////////////////////////////
_PUBLIC void siin_machine_info(_INOUT JSON_VALUE *info)
{
	char mac[20];

	if ((info == NULL) || (json_get_type(info) != JSON_OBJECT_TYPE))
		return;

	siin_mac(mac, sizeof(mac));
	json_object_add(info, "macaddress", json_string_new(mac));

	json_object_add(info, "unixtime", json_integer_new(time(NULL)));

#ifdef WIN32
	SYSTEM_INFO siSysInfo;
	char buf[PATH_LEN];
	char *aux;

	GetSystemInfo(&siSysInfo);

	json_object_add(info, "cores", json_integer_new(siSysInfo.dwNumberOfProcessors));

	switch(siSysInfo.wProcessorArchitecture)
	{
		case PROCESSOR_ARCHITECTURE_AMD64: aux = "x64 (AMD or Intel)"; break;
		case PROCESSOR_ARCHITECTURE_ARM: aux = "ARM"; break;
		case PROCESSOR_ARCHITECTURE_IA64: aux = "Intel Itanium-based"; break;
		case PROCESSOR_ARCHITECTURE_INTEL: aux = "x86"; break;
		default: aux = "desconhecido"; break;
	}

	json_object_add(info, "vendor", json_string_new(aux));

	switch (siSysInfo.dwProcessorType)
	{
		case PROCESSOR_INTEL_386: aux = "INTEL_386"; break;
		case PROCESSOR_INTEL_486: aux = "INTEL_486"; break;
		case PROCESSOR_INTEL_PENTIUM: aux = "INTEL_PENTIUM"; break;
		case PROCESSOR_INTEL_IA64: aux = "INTEL_IA64"; break;
		case PROCESSOR_AMD_X8664: aux = "AMD_X8664"; break;
		default: aux = "ARM(?)"; break;
	}

	json_object_add(info, "model", json_string_new(aux));

	OSVERSIONINFOEX versionInfo;
	versionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
	OSVERSIONINFO *pVersionInfo = (POSVERSIONINFO) &versionInfo;
	GetVersionEx(pVersionInfo);
	int max = (int) versionInfo.dwMajorVersion;
	int min = (int) versionInfo.dwMinorVersion;

	if ((max == 10) && (min == 0)) aux = "Windows 10 / Windows Server 2016";
	else if ((max == 6) && (min == 3)) aux = "Windows 8.1 / Windows Server 2012 R2";
	else if ((max == 6) && (min == 2)) aux = "Windows 8 / Windows Server 2012";
	else if ((max == 6) && (min == 1)) aux = "Windows 7 / Windows Server 2008 R2";
	else if ((max == 6) && (min == 0)) aux = "Windows Server 2008 / Windows Vista";
	else if ((max == 5) && (min == 2)) aux = "Windows Server 2003 R2 / Windows Server 2003";
	else if ((max == 5) && (min == 1)) aux = "Windows XP";
	else if ((max == 5) && (min == 0)) aux = "Windows 2000";
	else aux = "Windows > 10";

	snprintf(buf, PATH_LEN, "%s (%s)", aux, versionInfo.szCSDVersion);
	json_object_add(info, "so", json_string_new(buf));

	HW_PROFILE_INFO HwProfInfo;

	if (!GetCurrentHwProfile(&HwProfInfo)) 
	{
		fprintf(stderr, "GetCurrentHwProfile failed with error %lx\n", GetLastError());
	}
	else
	{
		aux = HwProfInfo.szHwProfileGuid;
		max = strlen(aux);
		aux[max - 1] = 0;
		json_object_add(info, "id", json_string_new(aux + 1));
	}

	WORD wVersionRequested;
	WSADATA wsaData;
	wVersionRequested = MAKEWORD(2, 2);
	WSAStartup(wVersionRequested, &wsaData);
	if (!gethostname(buf, sizeof(buf)))
		json_object_add(info, "hostname", json_string_new(buf));

#else
	siin_machine_info_proc(info);
	siin_machine_info_etc(info);
#endif

	JSON_VALUE *ifaces = json_array_new(1);
	json_object_add(info, "ifaces", ifaces);

	siin_get_ifaces(siin_machine_info_cb, ifaces);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int siin_hexlog_ascii(unsigned char *buf, char *log, int qtd)
{
	char *p = log;
	unsigned char c;
	int j;

	for (j = 0, p = log ; j < qtd ; j++)
	{
		c = buf[j];
		if ((c < ' ') || (c > '~'))
			p += snprintf(p, 4, ".");
		else
			p += snprintf(p, 4, "%c", c);
	}

	return(p - log);
}
// ////////////////////////////////////////////////////////////
_PUBLIC char * siin_hexlog(unsigned char *buf, int qtd)
{
	char *log = if_malloc(qtd * 10 + 16 * 4);
	char *p = log;
	int i, j, diff;

	if (qtd < 0)
	{
		log[0] = 0;
		return(log);
	}

	for (i = 0 ; i < qtd ; i++)
	{
		if ((i > 0) && ((i % 16) == 0))
		{
			p += snprintf(p, 10, "     ");
			p += siin_hexlog_ascii(buf + (i - 16), p, 16);
			p += snprintf(p, 4, "\n");
		}

		p += snprintf(p, 4, "%02X ", buf[i]);
	}

	diff = i % 16;
	if (diff == 0)
		diff = 16;

	for (j = 0 ; j < (16 - diff) ; j++)
		p += snprintf(p, 5, "   ");

	p += snprintf(p, 10, "     ");

	siin_hexlog_ascii(buf + (i - diff), p, diff);

	return(log);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
void * if_malloc(unsigned int size)
{
	void *m;

	// Malloc nao eh threadsafe no windows
#ifdef WIN32
	init_critical();
#endif
	m = malloc(size);

	mem_count++;
	if (m == NULL)
		verbose(stderr, "mem_count: %d   -   malloc(%d) = NULL;\n", mem_count, size);
	else
		memset(m, 0, size);
	
#ifdef DEBUG
	fprintf(stdout, "mem_count: %d   -   MALLOC(%X);\n", mem_count, (unsigned int) m);
	fflush(stdout);
#endif

	return(m);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
int if_free(void *m)
{
	mem_count--;
#ifdef DEBUG
	fprintf(stdout, "mem_count: %d   -   FREE(%X);\n", mem_count, (unsigned int) m);
	fflush(stdout);
#endif

#ifdef WIN32
	init_critical();
	free(m);
#else
	free(m);
#endif

	return(0);
}
// ////////////////////////////////////////////////////////////////////////




// ////////////////////////////////////////////////////////////////////////
void if_sleep(int t)
{
#ifdef WIN32
	Sleep(t);
#else
	usleep(t*1000);
#endif
}
// ////////////////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////////////////
int tokenizer(char s, char *str, char **tk, int max)
{
	int len;
	char *p = str;

	for (len = 1, tk[0] = p ; (*p != 0) && (len < max) ; p++)
	{
		if ((*p == '\n') || (*p == '\r'))
		{
			*p = 0;
		}

		if (*p == s)
		{
			*p = 0;
			tk[len++] = p + 1;
		}
	}

	return(len);
}
// ////////////////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////////////////// //
char * char2http(char *buf)
{
	int size = 3 * (strlen(buf) + 1);
	char *ret = (char *) if_malloc(size);
	unsigned char ms, ls;
	char *p, *q;

	memset(ret, 0, size);
	for (p = buf, q = ret ; *p != 0 ; p++)
	{
		if (((*p >= 'A') && (*p <= 'Z')) || 
			((*p >= 'a') && (*p <= 'z')) || ((*p >= '0') && (*p <= '9')))
		{
			*q++ = *p;
			*q = 0;
			continue;
		}

		*q++ = '%';
		ms = ((*p) >> 4) & 0x0F;
		ls = *p & 0x0F;
		if (ms > 9) ms = ms - 10 + 'A'; else ms += '0';
		if (ls > 9) ls = ls - 10 + 'A'; else ls += '0';
		*q++ = (char) ms;
		*q++ = (char) ls;
		*q = 0;
	}

	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int if_set_time(
	unsigned char year,
	unsigned char mon,
	unsigned char day,
	unsigned char hour,
	unsigned char min,
	unsigned char sec)
{
#ifdef WIN32
	SYSTEMTIME  st;
	
	GetLocalTime(&st);

	if (	(st.wYear == day) &&
		(st.wMonth == mon) &&
		(st.wDay == year) &&
		(st.wHour == hour) &&
		(st.wMinute == min) &&
		(st.wSecond == sec)	)
		return(0);

	 st.wYear = day;
	 st.wMonth = mon;
	 st.wDay = year;
	 st.wHour = hour;
	 st.wMinute = min;
	 st.wSecond = sec;
	
	SetLocalTime(&st);
#endif
#ifdef LINUX
	struct rtc_time rt;
	int fd, r;

	fd = open(DEV_RTC, O_RDONLY);
	if (fd < 0)
	{
		fprintf(stderr, "Erro ao tentar acessar: %s\n", DEV_RTC);
		return(1);
	}

	r = ioctl(fd, RTC_RD_TIME, &rt);
	if (r < 0)
	{
		fprintf(stderr, "Erro ao tentar ler horario do sistema.\n");
		return(2);
	}
	
	if (	(rt.tm_mday == day) &&
		(rt.tm_mon == mon) &&
		(rt.tm_year == year) && 
		(rt.tm_hour == hour) && 
		(rt.tm_min == min) && 
		(rt.tm_sec == sec)	)
		return(0);

	verbose(stdout, "Antes: %02d/%02d/%04d %02d:%02d:%02d\n",
		rt.tm_mday, rt.tm_mon + 1, rt.tm_year + 1900,
		rt.tm_hour, rt.tm_min, rt.tm_sec);
	
	verbose(stdout, "Depois: %02d/%02d/%04d %02d:%02d:%02d\n",
		day, mon + 1, year + 1900,
		hour, min, sec);

	rt.tm_mday = day;
	rt.tm_mon = mon;
	rt.tm_year = year;
	rt.tm_hour = hour;
	rt.tm_min = min;
	rt.tm_sec = sec;
	
	r = ioctl(fd, RTC_SET_TIME, &rt);
	if (r < 0)
	{
		fprintf(stderr, "Erro ao tentar alterar horario do sistema.\n");
		return(3);
	}
	close(fd);
#endif

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
IF_SLIST * if_slist_append(IF_SLIST *list, void *data)
{
	IF_SLIST *aux, *novo;

	novo = MALLOC(sizeof(IF_SLIST));
	if (novo == NULL)
	{
		verbose(stderr, "MALLOC ERROR...\n");
		return(list);
	}

	novo->data = data;
	novo->next = NULL;

	if (list == NULL)
		return(novo);

	// Vai ateh o fim
	for (aux = list ; aux->next != NULL ; aux = aux->next);

	aux->next = novo;

	return(list);
}
// ///////////////////////////////////////////////////////////////////// //
IF_SLIST * if_slist_prepend(IF_SLIST *list, void *data)
{
	IF_SLIST *novo;

	novo = MALLOC(sizeof(IF_SLIST));
	if (novo == NULL)
	{
		verbose(stderr, "MALLOC ERROR...\n");
		return(list);
	}

	novo->data = data;
	novo->next = list;

	return(novo);
}
// ///////////////////////////////////////////////////////////////////// //
unsigned int if_slist_length(IF_SLIST *list)
{
	unsigned int ret = 0;
	IF_SLIST *aux;
	
	for (aux = list, ret = 0 ; aux != NULL ; aux = aux->next, ret++);

	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //
IF_SLIST * if_slist_remove_data(IF_SLIST *list, void *data)
{
	IF_SLIST *aux, *del;

	if ((list == NULL) || (data == NULL))
		return(NULL);

	if (list->data == data)
	{
		aux = list->next;
		FREE(list);
		return(aux);
	}
	
	for (aux = list ; aux->next != NULL ; aux = aux->next)
	{
		if (aux->next->data != data)
			continue;

		del = aux->next;
		aux->next = aux->next->next;
		FREE(del);
		break;
	}

	return(list);
}
// ///////////////////////////////////////////////////////////////////// //
IF_SLIST * if_slist_remove_all(IF_SLIST *list)
{
	IF_SLIST *aux, *del;

	for (aux = list ; aux != NULL ; )
	{
		del = aux;
		aux = aux->next;
		FREE(del->data);
		FREE(del);
	}
	
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
IF_SLIST * if_slist_sort(IF_SLIST *list, IF_SLIST_SORT_FUNC func)
{
	IF_SLIST *first = list;
	IF_SLIST *aux1, *aux2, *ext, *p;
	int len, i;

	if (list == NULL)
		return(NULL);

	len = if_slist_length(list);
	for (i = 1 ; i < len ; i++)
	{
		aux1 = first->next;
		if (func(first->data, aux1->data) > 0)
		{
			aux2 = aux1->next;
			aux1->next = first;
			first->next = aux2;
			
			first = aux1;
		}

		for (ext = first, aux1 = ext->next ; aux1->next != NULL ; aux1 = aux1->next, ext = ext->next)
		{
			aux2 = aux1->next;
			if (func(aux1->data, aux2->data) > 0)
			{
				p = aux2->next;
				aux2->next = aux1;
				aux1->next = p;
				ext->next = aux2;

				aux1 = aux2;
			}
		}
	}

	return(first);
}
// ///////////////////////////////////////////////////////////////////// //


// Implementa if_db_xxxx
// ///////////////////////////////////////////////////////////////////// //
DLL_EXPORT char * if_db_get_value(IF_RESULTSET *rs, int row, char *field)
{
	IF_SLIST *aux;
	char **tuple;
	int col, i;

	if (rs == NULL)
		return(NULL);

	for (col = 0 ; rs->fields[col] != NULL ; col++)
		if (strcmp(rs->fields[col], field) == 0)
			break;

	if (col >= rs->num_fields)
		return(NULL);

	for (aux = rs->table, i = 0 ; (aux != NULL) && (i < row) ; aux = aux->next, i++)
		if (i == row)
			break;

	if (i != row)
		return(NULL);

	tuple = (char **) aux->data;

	return(tuple[col]);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int if_db_free_result(IF_RESULTSET *rs)
{
	IF_SLIST *aux;
	char **tuple;
	int i;

	if (rs == NULL)
		return(0);

	if (rs->json != NULL)
		FREE(rs->json);

	if (rs->sums != NULL)
		FREE(rs->sums);

	if (rs->sub_sums != NULL)
		FREE(rs->sub_sums);

	if (rs->query != NULL)
		FREE(rs->query);

	for (i = 0 ; rs->fields[i] != NULL ; i++)
		FREE(rs->fields[i]);
	FREE(rs->fields);

	for (aux = rs->table ; aux != NULL ; aux = aux->next)
	{
		tuple = (char **) aux->data;

		for (i = 0 ; tuple[i] != NULL ; i++)
			FREE(tuple[i]);
	}

	if_slist_remove_all(rs->table);
	FREE(rs);
	
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
long long if_atoll(_IN const char *str_n)
{
        long long nro = 0;
        int i;

        if (str_n == NULL)
                return(0);

        for (i = 0 ; str_n[i] == '0' ; i++)
                ;

        do
        {
                if ((str_n[i] < '0') || (str_n[i] > '9'))
                        break;

                nro = 10 * nro + (str_n[i] - '0');
        } while (str_n[++i] != 0);

        return(nro);
}
// ///////////////////////////////////////////////////////////////////// //



// ///////////////////////////////////////////////////////////////////// //
int if_encode(char *in, char out[100])
{
	static int seq[] = {11, 2097151, 1023, 101, 8191};
	int hash, i;
	
	for (i = 0, hash = 0 ; in[i] != 0 ; i++)
		hash += seq[i % 5] * in[i];

	return(sprintf(out, "%d", hash));
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int isNumber(char *str)
{
	int i;

	if (str == NULL)
		return(0);

	for (i = 0 ; str[i] != 0 ; i++)
	{
		if ((str[i] == '.') || ((str[i] >= '0') && (str[i] <= '9')))
			continue;

		return(0); 
	}

	return(1);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
BOM_CODE * if_get_BOM(char *txt)
{
	static BOM_CODE table[] = {
		{"UTF-8", {0xEF, 0xBB, 0xBF}, 3},
		{"UTF-16(BE)", {0xFE, 0xFF}, 2},
		{"UTF-16(LE)", {0xFF, 0xFE}, 2},
		{"UTF-32(BE)", {0x00, 0x00, 0xFE, 0xFF}, 4},
		{"UTF-32(LE)", {0xFF, 0xFE, 0x00, 0x00}, 4},
		{"UTF-7", {0x2B, 0x2F, 0x76, 0x38}, 4},
		{"UTF-1", {0xF7, 0x64, 0x4C}, 3},
		{"UTF-EBCDIC", {0xDD, 0x73, 0x66, 0x73}, 4},
		{"SCSU", {0x0E, 0xFE, 0xFF}, 3},
		{"BOCU-1", {0xFB, 0xEE, 0x28}, 3},
		{"GB-18030", {0x84, 0x31, 0x95, 0x33}, 4},
		{NULL, {0,0,0,0}, 0}
	};
	int i, k;

	for (i = 0 ; table[i].type != NULL ; i++)
	{
		for (k = 0 ; k < table[i].len ; k++)
			if (((unsigned char) txt[k]) != table[i].code[k])
				break;

		if (k == table[i].len)
			return(&(table[i]));
	}

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int get_config_ini(char *file, char **att_in, char **att_out)
{
	FILE *fd;
	char buf[BUFFER_LEN];
	int i, len;
	char *q;

	fd = fopen(file, "rb");
	if (fd == NULL)
		return(1);

	while (fgets(buf, BUFFER_LEN, fd) != NULL)
	{
		// limpa finalizadores '\r' '\n'
		for (i = 0 ; buf[i] != 0 ; i++)
			if ((buf[i] == '\r') || (buf[i] == '\n'))
				buf[i] = 0;
				
		for (i = 0 ; att_in[i] != NULL ; i++)
		{
			if (att_out[i] == NULL)
				att_out[i] = "";

			len = strlen(att_in[i]);
			if (strncmp(buf, att_in[i], len) == 0)
				if (buf[len] == '=')
				{
					q = buf + len + 1;
					att_out[i] = if_strdup(q);
				}
		}
	}
	
	fclose(fd);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int purgeZeros(char *str)
{
	char *p, *q;
	int n;

	if(str[0] != '0')
		return(0);

	for (p = str, n = 0 ; *p == '0' ; p++, n++)
		;

	for (q = str ; (*q = *p) ; p++, q++)
		;

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int siin_findKey(_IN char *properties[], _IN char *key)
{
	int i, len;

	if (properties == NULL)
		return(-1);

	for (len = 0 ; properties[len] != NULL ; len++)
		;

	for (i = 0 ; (i * 2 + 1) < len ; i++)
	{
		if (strcmp(key, properties[i * 2]) == 0)
			return(i * 2 + 1);
	}

	return(-2);
}
// ////////////////////////////////////////////////////////////
char * siin_getParam(_IN char *properties[], _IN char *key)
{
	int index = siin_findKey(properties, key);

	if (index < 0)
		return("");

	return(properties[index]);
}
// ///////////////////////////////////////////////////////////////////// //


// ////////////////////////////////////////////////////////////
int if_libloader(char *filename, LIB_API *api)
{
#ifdef WIN32
	if ((api->handle = LoadLibrary(filename)) == NULL) 
	{
		verbose(stderr, "Erro (%d) ao tentar carregar (%s)\n", GetLastError(), filename);
		return(2);
	}
#else
	if ((api->handle = dlopen(filename, RTLD_LAZY)) == NULL)
	{
		verbose(stderr, "%s\n", dlerror());
		fflush(stderr);
		return(2);
	}
#endif

	void *aux;
	for (int i = 0 ; api->bind[i].name != NULL ; i++)
	{
#ifdef WIN32
		aux = (void *) GetProcAddress(api->handle, (char *) api->bind[i].name);
#else
		aux = (void *) dlsym(api->handle, (char *) api->bind[i].name);
#endif
		if (aux == NULL)
		{
			verbose(stderr, "Metodo '%s' nao localizado em '%s'.\n", 
				api->bind[i].name, filename);
			return(3);
		}

		*(api->bind[i].func) = (void (*)()) aux;
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
