/****************************************************************************
 * Define as "classes":
 * 	IF_SLIST	- Lista encadeada
 *
****************************************************************************/

/** @file ifractal_utils.h Biblioteca de Suporte para iFractal Framework
\ingroup 

\note -----
\author Felipe Peressoni Waltrick	felipe@ifractal.com.br
\version 2.0
**/


#ifndef IFRACTAL_UTILS
#define IFRACTAL_UTILS

#ifdef WIN32

#include <winsock2.h>
#include <windows.h>
#include <ws2tcpip.h>
#include <process.h>
#include <iphlpapi.h>

#else

#include <sys/statfs.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <dlfcn.h>

#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdint.h>


#ifdef WIN32
#define SNPRINTF _snprintf
#else
#define SNPRINTF snprintf	/*!< Redefinicao da funcao "snprintf", _snprintf para WIN32 API. */
#include <pthread.h>
#endif

#define _DEPRECATED

#define MALLOC		if_malloc
#define STRDUP		if_strdup
#define FREE		if_free

#ifdef MSVC
#define DLL_EXPORT __declspec(dllexport)
#define DLL_IMPORT __declspec(dllimport)
#else
#define DLL_EXPORT		/*!< Diretiva de compilacao com MSVC para exportacao de metodos */
#define DLL_IMPORT		/*!< Diretiva de compilacao com MSVC para importacao de metodos */
#endif



/**
 * Enumeração que determina o tipo de requisição.
**/
typedef enum
{
	IF_HTTP_GET,
	IF_HTTP_POST,
	IF_HTTP_FILE,
	IF_HTTP_CMD_LEN
} IF_HTTP_CMD;
	

#define IP_LEN				64
#define BUFFER_LEN			4096
#define PATH_LEN			128

#define HOST_LEN			128
#define PORT_LEN			8
#define NAME_LEN			64
#define USER_LEN			64

#define DEV_RTC				"/dev/rtc"
#define LOG_LIMIT			50*1024*1024


// TODO - funcoes genericas, devem ser colocadas em uma biblioteca de uso geral
#define HTTP_HEADER_LENGTH     "Content-Length: "
#define HTTP_TRANSFER_ENCODING "Transfer-Encoding: chunked"
#define HTTP_GET_HEADERS       "GET /%s HTTP/1.1\r\nHost: %s\r\n\r\n"
#define HTTP_GET_HEADERS_AUTH  "GET /%s HTTP/1.1\r\nHost: %s\r\nAuthorization: Basic %s\r\n\r\n"
#define HTTP_POST_HEADERS      "POST /%s HTTP/1.1\r\nHost: %s\r\nContent-Type: application/x-www-form-urlencoded\r\nContent-Length: %d\r\n\r\n"

#define HTTP_FILE_HEADERS      "POST /%s HTTP/1.1\r\n" \
                               "Host: %s\r\n" \
                               "User-Agent: iFractal MODWEB/2.0\r\n" \
                               "Accept: text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.1\r\n" \
                               "Accept-Language: pt-br,en;q=0.7,en-us;q=0.3\r\n" \
                               "Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7\r\n" \
                               "Keep-Alive: 300\n" \
                               "Connection: keep-alive\n" \
                               "Content-Type: multipart/form-data; boundary=%s\r\n" \
                               "Content-Length: %d\r\n\r\n"


#include <url.h>


typedef struct
{
	int err_id;
	char *static_msg;
} IF_MSG_ERRO;


/**
Define uma lista encadeada simples.
**/
typedef struct _IF_SLIST
{
	void *data;
	struct _IF_SLIST *next;
} IF_SLIST;

/**
Acrescenta o elemento no fim da lista, o conteudo de data deve ser alocado com "malloc", 
caso contrario as funções "remove_x" não podem ser usadas.
**/
IF_SLIST * __attribute__ ((deprecated)) if_slist_append(IF_SLIST *, void *data);
/**
Acrescenta o elemento no inicio da lista.
**/
IF_SLIST * __attribute__ ((deprecated)) if_slist_prepend(IF_SLIST *, void *data);
unsigned int if_slist_length(IF_SLIST *);

/**
Remove o elemento e faz free em "data"
**/
IF_SLIST * __attribute__ ((deprecated)) if_slist_remove_data(IF_SLIST *, void *data);
IF_SLIST * __attribute__ ((deprecated)) if_slist_remove_all(IF_SLIST *);


/**
Prototipo para funções de ordenação.
**/
typedef int (*IF_SLIST_SORT_FUNC)(void *, void *);

IF_SLIST * __attribute__ ((deprecated)) if_slist_sort(IF_SLIST *, IF_SLIST_SORT_FUNC);



// Interface de acesso a banco de dados /////////////////
/**
 * Contem dados sobre o resultado de uma operacao de consulta:
 * 	Resultado da chamada do metodo "if_db_query"
**/
typedef struct _IF_RESULTSET
{
	int num_fields;		/*!< Quantidade de colunas */
	int num_rows;		/*!< Total de registros armazenados, limitados por "offset" e "limit" */
	int total_num_rows;	/*!< Total de registros retornados */
	char **fields;		/*!< Array com o nome das colunas */
	IF_SLIST *table;	/*!< Lista encadeada onde: Cada posição contem um vetor com valores das colunas. */
	int offset;		/*!< Offset da busca. */
	int limit;		/*!< Quantidade maxima de registros. */
	double *sums;		/*!< Vetor com o somatorio de toda a busca */
	double *sub_sums;	/*!< Vetor com o somatorio parcial */
	char *query;		/*!< Query original */
	char *json;		/*!< Json gerado por: "if_db_rs2json" */
} IF_RESULTSET;

/**
 * Contem dados relacionados a uma conexao/sessao com banco de dados. 
 * 	Resultado da chamada do metodo "if_db_connect".
**/
typedef struct _IF_DB
{
	void * handle;			/*!< Salva o descritor da API original */
	char host[HOST_LEN];		/*!< IP/Host da conexao. */
	char port[PORT_LEN];		/*!< Porta da conexao. */
	char user[NAME_LEN];
	char password[NAME_LEN];
	char db[NAME_LEN];
	char schema[NAME_LEN];
} IF_DB;


typedef enum
{
	IF_VERB_NONE,
	IF_VERB_FATAL,
	IF_VERB_ERROR,
	IF_VERB_WARN,
	IF_VERB_INFO,
	IF_VERB_DEBUG,
	IF_VERB_LEN
} IF_VERBOSITY;

#define VERBOSITY_OPTIONS		"{'help':'Nivel de verbosidade do log'," \
					"'options':[" \
					"{'text':'Silencioso', 'value':'0'}," \
					"{'text':'Erro Fatal', 'value':'1'}," \
					"{'text':'Erro', 'value':'2'}," \
					"{'text':'Advertencia', 'value':'3'}," \
					"{'text':'Informativo', 'value':'4'}," \
					"{'text':'Debug', 'value':'5'}]}"

/**
Função para compatibilizar Linux e WIN32.<BR/>
paramentro deve ser especificado em milisegundos.
**/
void if_sleep(int);	// em milisegundos

char * if_strdup(char *s);
void * if_malloc(unsigned int size);
int if_free(void *);

/**
Retorna string codificada em out.
**/
int if_encode(char *in, char out[100]);

int if_set_time(
	unsigned char year,
	unsigned char mon,
	unsigned char day,
	unsigned char hour,
	unsigned char min,
	unsigned char sec);

int if_init_log(char *conf_file);

#ifndef MSVC
long long if_atoll(_IN const char *);
#endif

#ifdef WIN32
__declspec(dllexport) int __stdcall verbose_base(char *msg);
#else
int verbose_base(char *msg);
#endif

int verbose(FILE *, char *format, ...);
int verbose2(IF_VERBOSITY vref, IF_VERBOSITY vdef, FILE *fd, char *source, char *format, ...);
int tokenizer(char s, char *str, char **tk, int max);


char * char2http(char *buf);


/**
Faz broadcast para auto reconhecimento dos modulos iFractal.
**/
int if_broadcast(char *port, void *data, int len);


typedef struct
{
	unsigned char id_B;
	unsigned char id_M;
} IF_BMP_SIGNATURE;

typedef struct
{
	int file_length;
	int reservado;		// Sempre 0
	int offset;
} IF_BMP_HEADER;


typedef struct
{
	int len;		// Sempre 0x28
	int width;
	int height;
	char planes[2];		// Sempre 0x0100
	char bpp[2];		// bits por pixel
	int compress;		// 0 = sem compressao
	int image_length;	// tamanho da area de dados
	int x_ppm;		// resolucao horizontal (pixels por metro)
	int y_ppm;		// resolucao vertical (pixels por metro)
	int clr_used;		// Numero de cores usadas na imagem
	int clr_real;		// Cores relevantes
} IF_BMP_INFO;


typedef struct
{
	unsigned char blue;
	unsigned char green;
	unsigned char red;
	unsigned char reserved;
} IF_BMP_PALLETE;



// iFractal Getopt
typedef enum
{
	IF_GETOPT_TYPE_NONE,
	IF_GETOPT_TYPE_STRING,
	IF_GETOPT_TYPE_NUMBER,
	IF_GETOPT_TYPE_HIDDEN,
	IF_GETOPT_TYPE_SELECT,
	IF_GETOPT_TYPE_BOOLEAN,
} IF_GETOPT_TYPE;


typedef struct
{
	char *value;
	char *text;
} IF_GETOPT_DOMAIN;

typedef struct
{
	_OUT int index;
	_IN char opt;
	_IN IF_GETOPT_TYPE type;
	_IN char *long_opt;
	_OUT char *arg;
	_OUT int argindex;
	_IN char *help;
	_IN IF_GETOPT_DOMAIN *domain;
} IF_GETOPT;


int if_getopt_key_in_file(char *file, char *key, _OUT char *value, size_t len);
int if_getopt_ini(_IN char *file, _INOUT IF_GETOPT opts[]);
int if_getopt_save(_IN char *file, _INOUT IF_GETOPT opts[]);
int if_getopt_sync(_IN IF_GETOPT in[], _INOUT IF_GETOPT out[]);
int if_getopt(_INOUT IF_GETOPT opts[], _IN int argc, _IN char **argv);
void if_getopt_verbose(_IN IF_GETOPT opts[]);
void if_getopt_help(_IN IF_GETOPT opts[]);
int if_getopt_isChecked(_IN IF_GETOPT opts[], _IN char *long_opt);
void if_getopt_check(_IN IF_GETOPT opts[], _IN char *long_opt);
char * if_getopt_getValue(_IN IF_GETOPT opts[], _IN char *long_opt);
int if_getopt_setValue(_IN IF_GETOPT opts[], _IN char *long_opt, _IN char *value);
int if_getopt_getIndex(_IN IF_GETOPT opts[], _IN char *long_opt);

unsigned char * hex2bin(_IN char *);
char * bin2hex(_IN unsigned char *, size_t len);
int isNumber(_IN char *);

typedef struct
{
	char *type;
	unsigned char code[4];
	size_t len;
} BOM_CODE;

BOM_CODE * if_get_BOM(char *);


int base64_get_encodedLen(_IN int length);
int b64_encode(_IN const unsigned char *inbuf, _IN size_t length, _OUT char **outbuf);
int b64_decode(_IN char *in64, _OUT unsigned char **outdata);

int qp_encode(_IN const unsigned char *inbuf, _IN size_t length, _OUT char **outbuf);
int qp_decode(_IN const char *inbuf, _IN size_t length, _OUT unsigned char **outbuf);

typedef int (*DELETE_CALLBACK)(char *path, char *filename, void *user_data);

typedef struct
{
	char *prefixo;
	char *sufixo;
	int qtdias;
	int recursivo;
	DELETE_CALLBACK callback;
	void *user_data;
	int apagados;
	int test;
} DELETE_CTX;

DELETE_CTX * delete_new(char *dirnome, char *prefix, char *suffix, int days, int rec, DELETE_CALLBACK cb, void *user_data, int test);
void delete_free(DELETE_CTX *);
int delete_perform(DELETE_CTX *, char *dir);

int execute_delete(char *dirnome, char *prefixo, char *sufixo, int qtdias, int recursivo, DELETE_CALLBACK, void *user_data);
int execute_delete_in(char *dirnome, DELETE_CTX *);
int delete_get_new_old(_IN char *dirnome, _IN char *prefixo, _IN char *sufixo, _OUT time_t *newer, _OUT time_t *older);

int purgeZeros(char *);


typedef struct
{
	void (**func)();
	char *name;
} LIB_BIND;


typedef struct
{
#ifdef WIN32
	HINSTANCE handle;
#else
	void *handle;
#endif
	LIB_BIND *bind;
} LIB_API;

int if_libloader(char *libfilename, LIB_API *);

#endif

