#ifndef PROC_MANAGER_H
#define PROC_MANAGER_H

#ifdef WIN32
#include <windows.h>
#else
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef __linux__
#include <wait.h>
#elif __APPLE__
#include <sys/wait.h>
#endif
#endif

#include <ifractal_threads.h>
#include <json.h>
#include <if_vetor.h>


typedef struct _PROCESS_CONTEXT PROCESS_CONTEXT;

typedef struct
{
	time_t unixtime;
	char *line;
} PROCESS_LOG;

typedef struct
{
#ifdef WIN32
	HANDLE pid;
#else
	int pid;
#endif
	int pin[2];
	int pout[2];
	int perr[2];
	time_t last_log;
	char **params;
	char **args;
	char *path;
	char **env;
	JSON_VALUE *status;
	VETOR *log;
	VETOR *logerr;
	THREAD_STATE state;
	unsigned char auto_clean;
} PROCESS;


typedef int (*PROCESS_START_CALLBACK)(PROCESS *, void *user_data);
typedef int (*PROCESS_END_CALLBACK)(PROCESS *, void *user_data);
typedef THREAD_STATE (*PROCESS_IDLE_CALLBACK)(PROCESS_CONTEXT *);
typedef int (*PROCESS_LOG_CALLBACK)(PROCESS *, char *line, void *user_data);
typedef int (*PROCESS_LOGERR_CALLBACK)(PROCESS *, char *line, void *user_data);
typedef int (*PROCESS_CLEAN_CALLBACK)(PROCESS *, void *user_data);

typedef struct
{	
	PROCESS_START_CALLBACK start;
	PROCESS_END_CALLBACK end;
	PROCESS_IDLE_CALLBACK idle;
	PROCESS_LOG_CALLBACK log;
	PROCESS_LOGERR_CALLBACK logerr;
	PROCESS_CLEAN_CALLBACK clean;
} PROCESS_CALLBACKS;

struct _PROCESS_CONTEXT
{
	VETOR *procs;
	PROCESS_CALLBACKS callbacks;
	IF_GETOPT *opts;
	void *user_data;
};


PROCESS_CONTEXT * process_context_new(PROCESS_CALLBACKS, IF_GETOPT *, void *user_data);
void process_context_free(PROCESS_CONTEXT *);
PROCESS * process_add(PROCESS_CONTEXT *, char **params, char **args, char *path, char **env);
int process_remove(PROCESS_CONTEXT *, PROCESS *);

int process_update(PROCESS *, char **params, char **args, char *path, char **env);

typedef int (*PROCESS_SELECT_CALLBACK)(PROCESS *, char **params, void *user_data);
int process_select_where_params(PROCESS_CONTEXT *, PROCESS_SELECT_CALLBACK, char **params, void *user_data);

int process_run(PROCESS_CONTEXT *);
int process_start_all(PROCESS_CONTEXT *);
int process_restart_all(PROCESS_CONTEXT *);
int process_kill_all(PROCESS_CONTEXT *);
int process_kill(PROCESS *);
void process_set_auto_clean(PROCESS *);

#ifdef WIN32
HANDLE process_start(PROCESS *);
#else
int process_start(PROCESS *);
#endif

int process_writeln_all(PROCESS_CONTEXT *, char *line);
int process_writeln(PROCESS *, char *line);
char * process_get_param(char **params, char *key);

#endif
