#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ifractal_utils.h>
#include <if_string.h>
#include <json.h>

#include <parser.h>



// ////////////////////////////////////////////////////////////
int isValidChar(char c)
{
	if ((c < '0') || (c > 'z'))
		return(0);

	if ((c >= 'A') && (c <= 'Z'))
		return(1);

	if ((c <= '9') || (c >= 'a') || (c == '_'))
		return(1);

	return(0);
}
// ////////////////////////////////////////////////////////////
char * next_field(_IN char *content, _IN char **position)
{
	char *last_field = *position;
	char aux[PATH_LEN];
	char *p;
	int i;

	if (last_field == NULL)
		p = content;
	else
		p = last_field++;

	if (*p == 0)
		return(NULL);

	for ( ; *p != 0 ; p++)
	{
		if (*p != '$')
			continue;

		p++;
		if (*p == 0)
			break;

		if ((*p >= '0') && (*p <= '9'))
			continue;

		for (i = 0 ; isValidChar(p[i]) ; i++)
		{
			aux[i] = p[i];
			aux[i + 1] = 0;
		}

		if ((i > 0) && ((aux[0] < '0') || (aux[0] > '9')))
		{
			*position = p + i;
			return(if_strdup(aux));
		}
	}

	return(NULL);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int dup_iter(JSON_VALUE *data_source, void *user_data)
{
	char *field = (char *) user_data;
	JSON_VALUE *name;

	name = json_object_find(data_source, "name");
	if (strcmp(field, json_get_string(name)) == 0)
		return(1);

	return(0);
}
// ////////////////////////////////////////////////////////////
int check_duplicatation(JSON_VALUE *data_sources, char *field)
{
	int q;

	q = json_array_iter(data_sources, dup_iter, field);
	if (q < json_array_length(data_sources))
		return(1);

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void fill_fixes(_INOUT JSON_VALUE *tpl)
{
	int i;
	char *str_fixes[] = {
#ifdef CMA
		"header/ident", 
		"header/envio_formato", 
		"header/envio_modelo", 
		"header/envio_assunto", 
		"header/envio_data_ini", 
		"header/envio_data_fim", 
		"header/envio_contato", 
		"header/envio_terminal", 
		"header/envio_cpf_cnpj", 
		"header/envio_contrato", 
		"header/envio_vencimento", 
#else
		"remetente",
		"email_remetente",
		"reply",
		"assunto",
		"modelo",
#endif
		NULL};

	char *source_list[] = {"name", "", NULL, NULL}; 
	char *token_list[] = {"value", "", "type", "FixedDataToken", NULL, NULL}; 
	JSON_VALUE *data_sources = json_object_find(tpl, "data_sources");
	JSON_VALUE *data_source, *data_tokens, *data_token;

	for (i = 0 ; str_fixes[i] != NULL ; i++)
	{
		data_tokens = json_array_new(2);
		data_token = json_object_new_list(token_list);
		json_array_add(data_tokens, data_token);

		source_list[1] = str_fixes[i];
		data_source = json_object_new_list(source_list);
		json_object_add(data_source, "data_tokens", data_tokens);
		json_array_add(data_sources, data_source);
	}
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * template_init()
{
	JSON_VALUE *tpl_init = json_parse_mem("{\"positional_line_keys\":[{\"initial_char\":\"1\",\"match\":\"1\"}],\"data_sources\":[]}");

	fill_fixes(tpl_init);

	return(tpl_init);
}
// ////////////////////////////////////////////////////////////
void generate_template(_IN char *content)
{
	JSON_VALUE *result = NULL;
	char *json, *field, *pos = NULL;
	char *source_list[] = {"name", "", NULL, NULL}; 
	char *token_list[] = {"type","LocalDataToken", "initial_char","", "content_length","", NULL,NULL}; 
	JSON_VALUE *data_sources, *data_source, *data_tokens, *data_token;

	result = template_init();
	data_sources = json_object_find(result, "data_sources");

	while ((field = next_field(content, &pos)) != NULL)
	{
		if (check_duplicatation(data_sources, field))
			continue;

		data_tokens = json_array_new(2);
		data_token = json_object_new_list(token_list);
		json_array_add(data_tokens, data_token);

		source_list[1] = field;
		data_source = json_object_new_list(source_list);
		json_object_add(data_source, "data_tokens", data_tokens);

		json_array_add(data_sources, data_source);

		if_free(field);
	}

	json = json_serialize(result);
	fprintf(stdout, "%s", json);
	if_free(json);

	json_value_free(result);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int find_value_iter(char *key, JSON_VALUE *obj, void *user_data)
{
	char **field_value = (char **) user_data;
	char *p;
	int i;

	// Procura '/'
	for (i = 0 ; (key[i] != 0) && (key[i] != '/') ; i++)
		;

	if (key[i] == '/')
		p = key + i + 1;
	else
		p = key;

	if (strcmp(p, field_value[0]) == 0)
	{
		field_value[1] = json_get_string(obj);
		return(1);
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
char * find_value(JSON_VALUE *globals, JSON_VALUE *record, char *field)
{
	char *field_value[] = {field, NULL};

	json_object_iter(record, find_value_iter, &field_value);
	if (field_value[1] == NULL)
		json_object_iter(globals, find_value_iter, &field_value);

	return(field_value[1]);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void print_interval(FILE *fd, char *ref, char *last)
{
	char *p;

	for (p = ref ; p < last ; p++)
		fprintf(fd, "%c", *p);
}
// ////////////////////////////////////////////////////////////
int process_record(JSON_VALUE *record, void *user_data)
{
	static int seq = 1;
	void **args = user_data;
	char *content = (char *) args[0];
	JSON_VALUE *globals = (JSON_VALUE *) args[1];
	char *path = (char *) args[2];
	char *field, *ref, *last, *value;
	char buf[PATH_LEN];
	FILE *fd;
	int len;

	snprintf(buf, PATH_LEN, "%s/PROC_%06d.html", path, seq++);
	fd = fopen(buf, "w");

	ref = content;
	last = NULL;
	while ((field = next_field(content, &last)) != NULL)
	{
		len = strlen(field) + 1;
		print_interval(fd, ref, last - len);
		ref = last;

		value = find_value(globals, record, field);
		if (value != NULL)
			fprintf(fd, "%s", value);

		if_free(field);
	}

	fprintf(fd, "%s", ref);
	fclose(fd);

	return(0);
}
// ////////////////////////////////////////////////////////////
void process_result(char *content, FILE *fd, char *path)
{
	JSON_VALUE *result = json_parse_file(fd);
	JSON_VALUE *globals, *records;
	void *args[3];

	if (result == NULL)
		goto process_result_error;

	globals = json_object_find(result, "globals");
	if (globals == NULL)
		goto process_result_error;

	records = json_object_find(result, "records");
	if (records == NULL)
		goto process_result_error;

	args[0] = content;
	args[1] = globals;
	args[2] = path;
	json_array_iter(records, process_record, args);

	return;

process_result_error:
	fprintf(stderr, "JSON invalido.\n");
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
char * processa_dados_modelo(char *globals_json, char *record_json, char *modelo_html)
{
	JSON_VALUE *globals = NULL, *record = NULL;
	char *ret;

	globals = json_parse_mem(globals_json);
	if (globals == NULL)
		globals = json_object_new(2);

	record = json_parse_mem(record_json);
	if (record == NULL)
		record = json_object_new(2);

	ret = process_model(globals, record, modelo_html);

	json_value_free(globals);
	json_value_free(record);

	return(ret);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * process_model(JSON_VALUE *globals, JSON_VALUE *record, char *modelo_html)
{
	char *field, *ref, *last, *value, *ret, *p, c[2] = {0,0};
	STRING_BUFFER *buffer;
	int len;

	buffer = string_new("");

	ref = modelo_html;
	last = NULL;
	while ((field = next_field(modelo_html, &last)) != NULL)
	{
		len = strlen(field) + 1;
		for (p = ref ; p < (last - len) ; p++)
		{
			c[0] = *p;
			string_append(buffer, c);
		}
		ref = last;

		value = find_value(globals, record, field);
		if (value != NULL)
			string_append(buffer, value);

		if_free(field);
	}

	string_append(buffer, ref);

	ret = if_strdup(string_get_text(buffer));
	string_free(buffer);

	return(ret);
}
// ////////////////////////////////////////////////////////////



#ifdef STANDALONE

// ////////////////////////////////////////////////////////////
FILE * getFDIN(char *file)
{
	FILE *fd;

	if ((file[0] == '-') && (file[1] == 0))
		return(stdin);

	fd = fopen(file, "rb");
	if (fd == NULL)
		fprintf(stderr, "Falha ao tentar abrir: '%s'.\n", file);

	return(fd);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int numera(char *prefixo)
{
	FILE *fd = stdin;
	int maxlen = 10 * BUFFER_LEN;
	char *buf = if_malloc(maxlen);
	int prefixo_len = strlen(prefixo);
	int seq = 1, linhas = 0;

	while (fgets(buf, maxlen, fd) != NULL)
	{
		if (strncmp(buf, prefixo, prefixo_len) != 0)
		{
			seq = 1;
			fprintf(stdout, "%s", buf);
			continue;
		}

		fprintf(stdout, "%s%02d%s", prefixo, seq++, buf + prefixo_len);		
		linhas++;
	}

	if_free(buf);

	return(linhas);
}
// ////////////////////////////////////////////////////////////



#define DESCRIPT	"Funcionalidades:\n" \
			"\t1) Gera template a partir de modelo.\n" \
			"\t2) Processa modelo a partir de arquivo resultado.\n" \
			"\t3) Acrescenta numeracao para linhas multiplas em arquivo de dados.\n"

static IF_GETOPT opts[] = {
	{0, 'n', IF_GETOPT_TYPE_STRING, "numera", "", 0, "Numera linhas multiplas conforme prefixo dado. (via stdin)"},
	{0, 'm', IF_GETOPT_TYPE_STRING, "modelo", "", 0, "Arquivo modelo."},
	{0, 'g', IF_GETOPT_TYPE_STRING, "test_globals", "", 0, "JSON com globais."},
	{0, 't', IF_GETOPT_TYPE_STRING, "test_record", "", 0, "JSON com registro."},
	{0, 'r', IF_GETOPT_TYPE_STRING, "result", "", 0, "Arquivo resultado (dados)."},
	{0, 'p', IF_GETOPT_TYPE_STRING, "path", ".", 0, "Path destino."},
	{0, 0, 0, 0, 0, 0, 0}
};


// ////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	char *globals_json, *record_json;
	char *modelo, *json, *content;
	FILE *fd;
	int r;

	r = if_getopt(opts, argc, argv);
	if ((r) || 
		(!if_getopt_isChecked(opts, "modelo") && !if_getopt_isChecked(opts, "numera"))
		)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(opts);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -m <MODELO>\n", argv[0]);

		fprintf(stderr, "\nExemplos:\n\tshell$ %s -m modelo.html > template.json\n", argv[0]);
		fprintf(stderr, "\tshell$ %s -m modelo.html -r result.json\n", argv[0]);
		fprintf(stderr, "\tshell$ %s -n 03 < dados.txt > dados_numerados.txt\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

	if (if_getopt_isChecked(opts, "numera"))
	{
		numera(if_getopt_getValue(opts, "numera"));
		return(0);
	}

	modelo = if_getopt_getValue(opts, "modelo");
	if ((fd = getFDIN(modelo)) == NULL)
	{
		fprintf(stderr, "Arquivo nao encontrado ou sem permissao: '%s'\n", modelo);
		return(1);
	}
	if_get_content(fd, &content);
	fclose(fd);

	if (if_getopt_isChecked(opts, "result"))
	{
		json = if_getopt_getValue(opts, "result");
		if ((fd = fopen(json, "r")) == NULL)
		{
			fprintf(stderr, "Arquivo de dados nao encontrado: '%s'\n", json);
			return(2);
		}
		process_result(content, fd, if_getopt_getValue(opts, "path"));
		fclose(fd);
	}
	else if ((if_getopt_isChecked(opts, "test_globals")) && (if_getopt_isChecked(opts, "test_record")))
	{
		globals_json = if_getopt_getValue(opts, "test_globals");
		record_json = if_getopt_getValue(opts, "test_record");
		modelo = processa_dados_modelo(globals_json, record_json, content);
		fprintf(stdout, "%s\n", modelo);
		if_free(modelo);
	}
	else
	{
		generate_template(content);
	}

	if_free(content);

	return(0);
}
// ////////////////////////////////////////////////////////////
#endif

