#include <ifractal.h>

#include <server.h>
#include <if_sched.h>


// Globais
extern IF_VERBOSITY verbosity;
extern IF_GETOPT configs[];


// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_set_cron(_INOUT SCHED *sched, _IN char *cron_string)
{
	char buf[PATH_LEN];
	char *cron[PERIOD_LEN + 1];
	int i, len;

	sched->period_type = SCHED_PERIOD_FIXED;

	strncpy(buf, cron_string, PATH_LEN);
	len = tokenizer(' ', buf, cron, PERIOD_LEN + 1);

	for (i = 0 ; i < PERIOD_LEN ; i++)
	{
		if ((i >= len) || (cron[i][0] == '*'))
		{
			sched->cron[i] = -1;
			continue;
		}

		sched->cron[i] = atoi(cron[i]);
	}

	return(0);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PRIVATE SCHED_PROC * sched_proc_new(_IN char *proc, _IN char *path, _IN char **args)
{
	SCHED_PROC *sched;
	int i;

	sched = if_malloc(sizeof(SCHED_PROC));
	memset(sched, 0, sizeof(SCHED_PROC));

	((SCHED *) sched)->type = SCHED_TYPE_PROC;

	strncpy(sched->proc, proc, PATH_LEN);
	strncpy(sched->path, path, PATH_LEN);

	sched->args[0] = if_strdup(proc);

	if (args != NULL)
	{
		for (i = 0 ; args[i] != NULL ; i++)
			sched->args[i + 1] = if_strdup(args[i]);
	}

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PRIVATE SCHED_URL * sched_url_new(_IN char *url, _IN char *path, _IN char **post_list)
{
	SCHED_URL *sched;
	int i;

	sched = if_malloc(sizeof(SCHED_URL));
	memset(sched, 0, sizeof(SCHED_URL));

	((SCHED *) sched)->type = SCHED_TYPE_URL;
	strncpy(sched->url, url, PATH_LEN);
	strncpy(sched->path, path, PATH_LEN);

	if (post_list != NULL)
	{
		for (i = 0 ; post_list[i] != NULL ; i++)
			sched->post_list[i] = if_strdup(post_list[i]);
	}
	else
	{
		sched->post_list[0] = NULL;
	}

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_proc_new_cron(_IN char *cron_string, _IN char *proc, _IN char *path, _IN char **args)
{
	SCHED *sched = (SCHED *) sched_proc_new(proc, path, args);
	sched_set_cron(sched, cron_string);

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_url_new_cron(_IN char *cron_string, _IN char *url, _IN char *path, _IN char **post_list)
{
	SCHED *sched = (SCHED *) sched_url_new(url, path, post_list);
	sched_set_cron(sched, cron_string);

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_delete_new(_IN char *path, _IN char *prefix, _IN char *suffix, _IN int days)
{
	SCHED_DELETE *sched;

	sched = if_malloc(sizeof(SCHED_DELETE));
	memset(sched, 0, sizeof(SCHED_DELETE));

	((SCHED *) sched)->type = SCHED_TYPE_DELETE;
	((SCHED *) sched)->period_type = SCHED_PERIOD_NONE;

	strncpy(sched->path, path, PATH_LEN);
	strncpy(sched->prefix, prefix, PATH_LEN);
	strncpy(sched->suffix, suffix, PATH_LEN);
	sched->days = days;

	return((SCHED *) sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_proc_new_freq(_IN int freq, _IN char *proc, _IN char *path, _IN char **args)
{
	SCHED *sched = (SCHED *) sched_proc_new(proc, path, args);
	sched->period_type = SCHED_PERIOD_FREQ;
	sched->frequency = freq;
	sched->last = 0;

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_url_new_freq(_IN int freq, _IN char *url, _IN char *path, _IN char **post_list)
{
	SCHED *sched = (SCHED *) sched_url_new(url, path, post_list);
	sched->period_type = SCHED_PERIOD_FREQ;
	sched->frequency = freq;
	sched->last = 0;

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC void sched_free(SCHED *sched)
{
	int i;

	if (sched == NULL)
		return;

	switch (sched->type)
	{
		case SCHED_TYPE_PROC:
			for (i = 0 ; ((SCHED_PROC *) sched)->args[i] != NULL ; i++)
				if_free(((SCHED_PROC *) sched)->args[i]);
			break;

		case SCHED_TYPE_URL:
			for (i = 0 ; ((SCHED_URL *) sched)->post_list[i] != NULL ; i++)
				if_free(((SCHED_URL *) sched)->post_list[i]);
			break;

		case SCHED_TYPE_UPLOAD:
			json_value_free(((SCHED_UPLOAD *) sched)->jconfig);
			break;

		default:
			break;
	}

	if_free(sched);
}
// ////////////////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_new_json_url_iter(_IN char *key, _IN JSON_VALUE *value, void *user_data)
{
	char **post_list = (char **) user_data;
	int i;

	for (i = 0 ; (post_list[i] != NULL) && (i < (PATH_LEN - 2)) ; i++)
		;

	post_list[i] = key;
	post_list[i + 1] = json_get_string(value);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_new_json_url(_IN JSON_VALUE *json, _IN char **params)
{
	char *cron, *url, *path;
	SCHED *sched = NULL;
	char **post_list = NULL;
	char *buf[PATH_LEN];
	JSON_VALUE *jpost;
	int i, freq;

	jpost = json_object_find(json, "post");
	if ((jpost != NULL) && (json_get_type(jpost) == JSON_OBJECT_TYPE))
	{
		for (i = 0 ; i < PATH_LEN ; i++)
			buf[i] = NULL;

		json_object_iter(jpost, sched_new_json_url_iter, buf);
		post_list = buf;
	}

	url = json_get_list_param(params, "url");
	path = json_get_list_param(params, "path");
	cron = json_get_list_param(params, "cron");
	if (cron[0] != 0)
	{
		sched = sched_url_new_cron(cron, url, path, post_list);
	}
	else
	{
		freq = atoi(json_get_list_param(params, "freq"));
		sched = sched_url_new_freq(freq, url, path, post_list);
	}

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_new_json_proc_iter(_IN JSON_VALUE *value, void *user_data)
{
	char **post_list = (char **) user_data;
	int i;

	for (i = 0 ; (post_list[i] != NULL) && (i < (PATH_LEN - 1)) ; i++)
		;

	post_list[i] = json_get_string(value);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_new_json_proc(_IN JSON_VALUE *json, char **params)
{
	char *cron, *proc, *path;
	SCHED *sched = NULL;
	char **args = NULL;
	char *buf[PATH_LEN];
	JSON_VALUE *jargs;
	int i, freq;

	jargs = json_object_find(json, "args");
	if ((jargs != NULL) && (json_get_type(jargs) == JSON_ARRAY_TYPE))
	{
		for (i = 0 ; i < PATH_LEN ; i++)
			buf[i] = NULL;

		json_array_iter(jargs, sched_new_json_proc_iter, buf);
		args = buf;
	}

	proc = json_get_list_param(params, "proc");
	path = json_get_list_param(params, "path");
	cron = json_get_list_param(params, "cron");
	if (cron[0] != 0)
	{
		sched = sched_proc_new_cron(cron, proc, path, args);
	}
	else
	{
		freq = atoi(json_get_list_param(params, "freq"));
		sched = sched_proc_new_freq(freq, proc, path, args);
	}

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_new_json_delete(_IN JSON_VALUE *json, char **params)
{
	char *path, *prefix, *suffix, *days;

	path = json_get_list_param(params, "path");
	prefix = json_get_list_param(params, "prefix");
	suffix = json_get_list_param(params, "suffix");
	days = json_get_list_param(params, "days");

	return(sched_delete_new(path, prefix, suffix, atoi(days)));
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
SCHED * sched_upload_new(_IN JSON_VALUE *jconfig)
{
	SCHED *sched = if_malloc(sizeof(SCHED_UPLOAD));
	JSON_VALUE *aux;
	int freq = 5;

	memset(sched, 0, sizeof(SCHED_UPLOAD));

	aux = json_object_find(jconfig, "freq");
	if (aux != NULL)
		freq = json_get_int(aux);

	sched->type = SCHED_TYPE_UPLOAD;
	sched->period_type = SCHED_PERIOD_FREQ;
	sched->frequency = freq;
	sched->last = 0;

	((SCHED_UPLOAD *) sched)->jconfig = json_clone(jconfig);

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_PUBLIC SCHED * sched_new_json(_IN JSON_VALUE *json)
{
	char *name, *proc, *url, *prefix, *suffix, *days, *inpath;
	char *params[] = {
		"name","sched",
		"inpath","",
		"path",".",
		"url","",
		"proc","",
		"freq","",
		"cron","",
		"prefix","",
		"suffix","",
		"days","1",
		NULL,NULL
		};
	SCHED *sched;

	if (json == NULL)
		return(NULL);

	json_make_list_param(json, params);

	proc = json_get_list_param(params, "proc");
	url = json_get_list_param(params, "url");
	inpath = json_get_list_param(params, "inpath");
	prefix = json_get_list_param(params, "prefix");
	suffix = json_get_list_param(params, "suffix");
	days = json_get_list_param(params, "days");

	if (proc[0] != 0)
		sched = sched_new_json_proc(json, params);
	else if (inpath[0] != 0)
		sched = sched_upload_new(json);
	else if (url[0] != 0)
		sched = sched_new_json_url(json, params);
	else if ((days[0] != 0) && ((prefix[0] != 0) || (suffix[0] != 0)))
		sched = sched_new_json_delete(json, params);
	else
		return(NULL);

	name = json_get_list_param(params, "name");
	strncpy(sched->name, name, PATH_LEN);

	return(sched);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_load_jobs_iter(JSON_VALUE *json, void *user_data)
{
	VETOR *vet = (VETOR *) user_data;
	SCHED *sched = sched_new_json(json);

	if (sched != NULL)
		vetor_push(vet, sched);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC VETOR * sched_load_jobs(char *config_json)
{
	VETOR *vet = NULL;
	JSON_VALUE *json;
	FILE *fd;

	fd = fopen(config_json, "r");
	if (fd == NULL)
		return(NULL);

	json = json_parse_file(fd);
	fclose(fd);

	if (json == NULL)
		return(NULL);

	vet = vetor_new();
	json_array_iter(json, sched_load_jobs_iter, vet);
	json_value_free(json);

	return(vet);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_verbose_iter(VETOR *jobs, int index, void *data, void *user_data)
{
	SCHED *sched = (SCHED *) data;
	FILE *fd = (FILE *) user_data;
	SCHED_DELETE *del;
	SCHED_PROC *proc;
	SCHED_URL *url;
	int i;

	verbose(fd, "\n");
	verbose(fd, "#################################################\n");
	verbose(fd, "name: '%s'\n", sched->name);

	switch (sched->type)
	{
		case SCHED_TYPE_PROC:
			proc = (SCHED_PROC *) sched;
			verbose(fd, "proc: '%s'\n", proc->proc);
			verbose(fd, "path: '%s'\n", proc->path);
			verbose(fd, "args:\n");
                        
			for (i = 0 ; proc->args[i] != NULL ; i++)
				verbose(fd, "\t'%s'\n", proc->args[i]);
			break;

		case SCHED_TYPE_URL:
			url = (SCHED_URL *) sched;
			verbose(fd, "url: '%s'\n", url->url);
			verbose(fd, "path: '%s'\n", url->path);
			verbose(fd, "post:\n");
                        
			for (i = 0 ; url->post_list[i] != NULL ; i += 2)
				verbose(fd, "\t'%s':'%s'\n", url->post_list[i], url->post_list[i + 1]);
			break;

		case SCHED_TYPE_DELETE:
			del = (SCHED_DELETE *) sched;
			verbose(fd, "path: '%s'  -  '%s'....'%s'  -  %d dia(s)\n", 
				del->path,
				del->prefix,
				del->suffix,
				del->days);
			break;

		case SCHED_TYPE_UPLOAD:
			break;

		case SCHED_TYPE_LEN:
			break;
	}

	switch (sched->period_type)
	{
		case SCHED_PERIOD_FREQ:
			verbose(fd, "freq: %d\n", sched->frequency);
			break;

		case SCHED_PERIOD_FIXED:
			verbose(fd, "cron:\n");
			verbose(fd, "\tmin: %d\n", sched->cron[0]);
			verbose(fd, "\thour: %d\n", sched->cron[1]);
			verbose(fd, "\tday: %d\n", sched->cron[2]);
			verbose(fd, "\tmonth: %d\n", sched->cron[3]);
			verbose(fd, "\tweek: %d\n", sched->cron[4]);
			break;

		case SCHED_PERIOD_NONE:
			break;
	}
	verbose(fd, "#################################################\n");

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC void sched_verbose(VETOR *jobs)
{
	vetor_iter(jobs, sched_verbose_iter, stdout);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_execute_upload(SCHED_UPLOAD *sched)
{
	int count = 0;

	modupload_perform(sched->jconfig, &count);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_execute_delete_iter(char *path, char *filename, void *user_data)
{
	verbose(stderr, "Delete: '%s'\n", filename);
	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_execute_delete(SCHED_DELETE *sched)
{
	int r;

	//verbose(stderr, "Executa delete: '%s' %d dia(s)\n", sched->path, sched->days);
	r = execute_delete(
		sched->path, 
		sched->prefix, 
		sched->suffix, 
		sched->days, 
		0, 
		sched_execute_delete_iter, 
		NULL);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_execute_proc(SCHED_PROC *sched)
{
#ifdef WIN32
	char cmd_path[BUFFER_LEN];
	char *p;
	int i;

	p = cmd_path;
	p += snprintf(p, PATH_LEN, "%s", sched->proc);
	for (i = 1 ; sched->args[i] != NULL ; i++)
		p += snprintf(p, PATH_LEN, " \"%s\"", sched->args[i]);

	verbose(stderr, "exec: '%s'\n", cmd_path);

	PROCESS_INFORMATION pi;
	STARTUPINFO si;
	BOOL ret;

	ZeroMemory( &si, sizeof(si) );
	si.cb = sizeof(si);
	ZeroMemory( &pi, sizeof(PROCESS_INFORMATION) );
        
	// Start the child process. 
	if( !(ret = CreateProcess(sched->args[0], cmd_path, NULL, NULL, FALSE, 0, NULL, sched->path, &si, &pi )) ) 
	{
		verbose(stderr, "%s\n", cmd_path);
		verbose(stderr, "CreateProcess failed (%d).\n", (int) GetLastError());
		return(-2);
	}
#else
	int pid = fork();
	if (pid < 0)
		return(-2);
	
	// Filho
	if (pid == 0)
	{
		// Posiciona o diretorio do processo
		if (sched->path != NULL)
		{
			if (chdir(sched->path))
			{
				fprintf(stderr, "Falha ao tentar ajustar path: '%s'\n", sched->path);
				exit(1);
			}
		}

		// executa o processo
		execv(sched->proc, sched->args);
		
		// Se a aplicacao existir e tudo der certo
		// esse codigo nunca serah executado
		fprintf(stderr, "Falha ao tentar executar: '%s'\n", sched->proc);
		exit(2);
	}
#endif

	return(0);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
_CALLBACK int sched_execute_url_iter(void *user_data, double dltotal, double dlnow, double ultotal, double ulnow)
{
	time_t *start = (time_t *) user_data;
	time_t diff = time(NULL) - *start;

	if ((diff > 300) && (ulnow == 0))
	{
		verbose(stderr, "URL timeout.\n");
		return(1);
	}

	//verbose(stderr, "Upload: %0.lf/%0.lf   -   Download: %0.lf/%0.lf (%d segs)\n",
	//	ulnow, ultotal, dlnow, dltotal, diff);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_execute_url_save(SCHED_URL *sched, URL *url, unsigned char *resp, int len)
{
	char *fopen_param = siin_getParam(sched->post_list, "fopen_param");
	char *out_filename = siin_getParam(sched->post_list, "out_filename");
	char filename[PATH_LEN];
	FILE *fd;

	if (out_filename[0] == 0)
	{
		if (url->filename[0] == 0)
			out_filename = url->file;
		else
			out_filename = url->filename;
	}

	if (fopen_param[0] == 0)
		fopen_param = "w";

	if ((sched->path[0] == 0) || (out_filename[0] == 0))
		return(-1);

	snprintf(filename, sizeof(filename), "%s%c%s", sched->path, PATH_SEPARATOR, out_filename);
	fd = fopen(filename, fopen_param);
	if (fd == NULL)
	{
		verbose(stderr, "Falha ao tentar salvar: '%s'\n", filename);
		return(-2);
	}

	fwrite(resp, 1, len, fd);

	fclose(fd);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
_PRIVATE int sched_execute_url(SCHED_URL *sched)
{
	unsigned char *resp = NULL;
	time_t now = time(NULL);
	URL *url;
	int r;

	url = url_new(sched->url);

	r = url_execute_post(url, sched->post_list, &resp, sched_execute_url_iter, &now);
	if (resp != NULL)
	{
		sched_execute_url_save(sched, url, resp, r);
		if_free(resp);
	}

	url_free(url);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC int sched_execute(SCHED *sched)
{
	switch (sched->type)
	{
		case SCHED_TYPE_PROC:
			sched_execute_proc((SCHED_PROC *) sched);
			break;

		case SCHED_TYPE_URL:
			sched_execute_url((SCHED_URL *) sched);
			break;

		case SCHED_TYPE_DELETE:
			sched_execute_delete((SCHED_DELETE *) sched);
			break;

		case SCHED_TYPE_UPLOAD:
			sched_execute_upload((SCHED_UPLOAD *) sched);
			break;

		case SCHED_TYPE_LEN:
			break;
	}

	sched->last = time(NULL);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////

// ///////////////////////////////////////////////////////////////////// //
_PUBLIC int sched_check(SCHED *sched)
{
	time_t now = time(NULL);
	int cron[PERIOD_LEN];
	struct tm *dt;
	int i;

	switch (sched->period_type)
	{
		case SCHED_PERIOD_FREQ:
			if (((now - sched->last) / 60) < sched->frequency)
				return(0);
			break;

		case SCHED_PERIOD_FIXED:
			dt = localtime(&now);
        
			cron[PERIOD_DWEEK] = dt->tm_wday;
			cron[PERIOD_MONTH] = dt->tm_mon;
			cron[PERIOD_DAY] = dt->tm_mday;
			cron[PERIOD_HOUR] = dt->tm_hour;
			cron[PERIOD_MIN] = dt->tm_min;
        
			for (i = 0 ; i < PERIOD_LEN ; i++)
			{
				//verbose(stderr, "(%d,%d)\n", sched->cron[i], cron[i]);
				if ((cron[i] != sched->cron[i]) && (sched->cron[i] >= 0))
					return(0);
			}

		case SCHED_PERIOD_NONE:
			if (((now - sched->last) / 60) < 1)
				return(0);
			break;
	}

	return(1);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
_PUBLIC int sched_clean(_IN VETOR *jobs)
{
#ifdef WIN32
	return(0);
#else
	int pid, status, n;

	for (n = 0 ; (pid = waitpid(0, &status, WNOHANG)) > 0 ; n++)
		verbose(stderr, "Finaliza: %d\n", pid);

	return(n);
#endif
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
_CALLBACK int modweb_sched_iter(VETOR *jobs, int index, void *data, void *user_data)
{
	SCHED *sched = (SCHED *) data;

	if (sched_check(sched))
		sched_execute(sched);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC int modweb_sched(MODSIIN_CONTEXT *ctx, THREAD_STATE (*idle)(MODSIIN_CONTEXT *, void *), void *user_data)
{
	VETOR *jobs = NULL;
	char *config;

	config = if_getopt_getValue(configs, "SCHEDCONFIG");

	jobs = sched_load_jobs(config);
	if (jobs == NULL)
	{
		verbose(stdout, "|SCHED| nenhum processo, URL, delete agendado.\n");
		return(-1);
	}

	sched_verbose(jobs);

	while (1)
	{
		if ((idle != NULL) && (idle(ctx, user_data) != TH_ALIVE))
			break;

		vetor_iter(jobs, modweb_sched_iter, NULL);
		if_sleep(2000);
		sched_clean(jobs);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //



#ifdef STANDALONE
#define DESCRIPT	"Agendamento de processos."

// //////////////////////////////////////////////////////////////////////
_PUBLIC int modweb_main(MODSIIN_CONTEXT *ctx, THREAD_STATE (*idle)(MODSIIN_CONTEXT *, void *), void *user_data)
{
	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_CALLBACK THREAD_STATE sched_idle(MODSIIN_CONTEXT *ctx, void *user_data)
{
	return(TH_ALIVE);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int main (int argc, char **argv)
{
	int r;
	
	r = if_getopt(configs, argc, argv);
	if (	(r) || 
		!if_getopt_isChecked(configs, "SCHEDCONFIG")
	)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -s <JSON_CONFIG>\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

	verbosity = atoi(if_getopt_getValue(configs, "VERBOSITY"));

	modweb_sched(NULL, sched_idle, NULL);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

#endif

