#include <ifractal.h>
#include <boleto_barras.h>

extern JSON_VALUE * parser_currency(
	char *, JSON_VALUE *, JSON_VALUE *, JSON_VALUE *, 
	JSON_VALUE *, JSON_VALUE *, JSON_VALUE *);
extern int core_purge_spaces(char *);

typedef struct
{
	char *name;
	int length;
} VOB_ARG;

VOB_ARG vob_args[] = {
	{NULL, 0},
	{"BLOCO1", 5},
	{"BLOCO2", 5},
	{"BLOCO3", 5},
	{"BLOCO4", 6},
	{"BLOCO5", 5},
	{"BLOCO6", 6},
	{"DAC", 1},
	{"VENCIMENTO/VALOR", 14}
};

int args_len = sizeof(vob_args) / sizeof(VOB_ARG);


typedef	struct
{
	char *name;
	int length;
	char *key;
} BOLETO_FIELD;

typedef struct
{
	char *name;
	char *id;
	BOLETO_FIELD *fields;
} BOLETO_BANCO;


BOLETO_FIELD bb_sr[] = {
	{"nosso_numero", 11, NULL},
	{"agencia", 4, NULL},
	{"fixo", 0, "000"},
	{"conta", 5, NULL},
	{"carteira", 2, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD bb[] = {
	{"convenio", 6, NULL},
	{"nosso_numero", 17, NULL},
	{"carteira", 2, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD bradesco[] = {
	{"agencia", 4, NULL},
	{"carteira", 2, NULL},
	{"nosso_numero", 11, NULL},
	{"conta", 7, NULL},
	{"fixo", 0, "0"},
	{NULL, 0, NULL}
};

BOLETO_FIELD itau198[] = {
	{"carteira", 0, "198"},
	{"nosso_numero", 8, NULL},
	{"nro_documento", 7, NULL},
	{"conta", 5, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD itau[] = {
	{"carteira", 3, NULL},
	{"nosso_numero", 8, NULL},
	{"dac", 1, NULL},
	{"agencia", 4, NULL},
	{"conta", 6, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD santander[] = {
	{"fixo", 0, "9"},
	{"cod_beneficiario", 7, NULL},
	{"nosso_numero", 12, NULL},
	{"dac", 1, NULL},
	{"fixo", 0, "0"},
	{"carteira", 3, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD cef[] = {
	{"nosso_numero", 10, NULL},
	{"agencia", 4, NULL},
	{"cod_beneficiario", 11, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD cef_sigcb[] = {
	{"cod_beneficiario", 6, NULL},
	{"dac_1", 1, NULL},
	{"nosso_numero_1", 3, NULL},
	{"carteira", 1, NULL},
	{"nosso_numero_2", 3, NULL},
	{"emissor", 1, NULL},
	{"nosso_numero_3", 9, NULL},
	{"dac_2", 1, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD hsbc[] = {
	{"cod_beneficiario", 7, NULL},
	{"nosso_numero", 13, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD pan[] = {
	{"agencia", 4, NULL},
	{"carteira", 3, NULL},
	{"cod_beneficiario", 7, NULL},
	{"nosso_numero", 10, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD citi[] = {
	{"produto", 1, NULL},
	{"carteira", 3, NULL},
	{"cod_beneficiario", 9, NULL},
	{"nosso_numero", 11, NULL},
	{NULL, 0, NULL}
};

BOLETO_FIELD bgn[] = {
	{"carteira", 2, NULL},
	{"cod_beneficiario", 3, NULL},
	{"nosso_numero", 13, NULL},
	{"parcela", 3, NULL},
	{"fixo", 0, "0000"},
	{NULL, 0, NULL}
};


BOLETO_BANCO vob_bancos[] = {
	{"Banco do Brasil", "001", bb},
	{"Banco do Brasil (sem registro)", "001", bb_sr},
	{"Bradesco", "237", bradesco},
	{"Itau (cart: 198)", "341", itau198},
	{"Itau", "341", itau},
	{"Santander", "033", santander},
	{"Caixa Economica Federal (sigcb)", "104", cef_sigcb},
	{"Caixa Economica Federal", "104", cef},
	{"HSBC", "399", hsbc},
	{"Banco Pan", "623", pan},
	{"Citibank", "745", citi},
	{"Banco BGN", "739", bgn},
	{NULL, NULL, NULL}
};



// ////////////////////////////////////////////////////////////
int vob_checkin(char **argv)
{
	char block[100], *dac, *p;
	int i, k;

	for (i = 1 ; i < args_len ; i++)
	{
		if (vob_args[i].length != strlen(argv[i]))
		{
			fprintf(stderr, "Erro: Quantidade de digitos invalida no campo '%s'.\n", vob_args[i].name);
			return(2);
		}
	}

	for (k = 1 ; k < 7 ; k += 2)
	{
		i = snprintf(block, sizeof(block), "%s%s", argv[k], argv[k + 1]);
		block[i - 1] = 0;
		i = strlen(argv[k + 1]) - 1;
		dac = barras_DAC10(block);
		if (dac[0] != argv[k + 1][i])
		{
			fprintf(stderr, "Erro digitacao: %s/%s\n", 
				vob_args[k].name, vob_args[k + 1].name);
			return(k);
		}

		if_free(dac);
	}

	p = block;
	p += snprintf(p, sizeof(block), "%c%c%c%c", argv[1][0], argv[1][1], argv[1][2], argv[1][3]);
	p += snprintf(p, sizeof(block), "%s%c", argv[8], argv[1][4]);

	// Elimina os DACs dos blocos
	for (k = 2 ; k < 7 ; k += 2)
	{
		i = strlen(argv[k]) - 1;
		argv[k][i] = 0;
	}

	p += snprintf(p, sizeof(block), "%s%s%s%s%s", argv[2], argv[3], argv[4], argv[5], argv[6]);

	dac = barras_DAC11(block);
	if (dac[0] != argv[7][0])
	{
		fprintf(stderr, "Erro digitacao: %s/%s\n",
			vob_args[7].name, vob_args[8].name);
		return(8);
	}

	if_free(dac);

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * vob_get_valor(JSON_VALUE *jfields, char *vvlinha)
{
	char valor[50];

	if (strncmp(vvlinha + 4, "0000000000", 10) == 0)
		return("");

	snprintf(valor, sizeof(valor), "%s", vvlinha + 4);
	parser_currency(valor, NULL, NULL, NULL, NULL, NULL, NULL);

	json_object_add(jfields, "valor", json_string_new(valor));

	return(json_object_get_string(jfields, "valor"));
}
// ////////////////////////////////////////////////////////////
char * vob_get_vencimento(JSON_VALUE *jfields, char *vvlinha)
{
	char vencimento[50];
	int fator, k, i;
	struct tm ref;
	struct tm *dt;
	time_t unixtime;

	if (strncmp(vvlinha, "0000", 4) == 0)
		return("");

	// 07 de Outubro de 1997
	memset(&ref, 0, sizeof(struct tm));
	ref.tm_mday = 7;
	ref.tm_mon = 10 - 1;
	ref.tm_year = 97;

	unixtime = mktime(&ref);

	for (fator = 0, i = 0, k = 1000 ; i < 4 ; i++, k /= 10)
		fator += (vvlinha[i] - '0') * k;

	unixtime += fator * 86400;

	dt = localtime(&unixtime);
	snprintf(vencimento, sizeof(vencimento), "%02d/%02d/%d", 
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900);

	json_object_add(jfields, "vencimento", json_string_new(vencimento));

	return(json_object_get_string(jfields, "vencimento"));
}
// ////////////////////////////////////////////////////////////
int vob_get_banco_check_fields(JSON_VALUE *jfields, int bi, char *block)
{
	int i, off, len = 0;
	char value[50];

	for (off = 0, i = 0 ; vob_bancos[bi].fields[i].name != NULL ; i++, off += len)
	{
		if (vob_bancos[bi].fields[i].key != NULL)
		{
			len = strlen(vob_bancos[bi].fields[i].key);
			if (strncmp(block + off, vob_bancos[bi].fields[i].key, len) != 0)
				return(1);
		}
		else
		{
			len = vob_bancos[bi].fields[i].length;
			memcpy(value, block + off, len);
			value[len] = 0;

			json_object_add(jfields, vob_bancos[bi].fields[i].name, json_string_new(value));
		}
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * vob_get_banco(char **argv)
{
	char *list[] = {"erro","0", "descricao","OK", NULL,NULL};
	JSON_VALUE *jfields = json_object_new_list(list);
	char block[100];
	int i;

	snprintf(block, sizeof(block), "%c%s%s%s%s%s",
		argv[1][4], argv[2], argv[3], argv[4], argv[5], argv[6]);

	//fprintf(stderr, "%s\n", block);

	for (i = 0 ; vob_bancos[i].name != NULL ; i++)
	{
		if (strncmp(vob_bancos[i].id, argv[1], 3) != 0)
			continue;

		if (!vob_get_banco_check_fields(jfields, i, block))
			break;
	}

	if (vob_bancos[i].name == NULL)
	{
		json_object_add(jfields, "erro", json_string_new("1"));
		json_object_add(jfields, "descricao", json_string_new("banco nao homologado"));
	}
	else
	{
		json_object_add(jfields, "banco", json_string_new(vob_bancos[i].id));
		json_object_add(jfields, "nome", json_string_new(vob_bancos[i].name));
	}

	vob_get_vencimento(jfields, argv[8]);
	vob_get_valor(jfields, argv[8]);

	return(jfields);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
_PUBLIC char * vob_decode(char *linha)
{
	int len, linha_len, i, off;
	char bm[args_len][20];
	char *blocos[args_len];
	JSON_VALUE *jfields; 
	char *buf;

	if (linha == NULL)
		return("{\"erro\":97}");

	buf = if_strdup(linha);

	for (i = 0, linha_len = 0 ; i < args_len ; i++)
		linha_len += vob_args[i].length;

	memset(bm, 0, sizeof(bm));
	for (i = 0 ; i < args_len ; i++)
		blocos[i] = bm[i];

	core_purge_spaces(buf);
	len = strlen(buf);
	if (len == linha_len)			// linha digitavel
	{
		for (i = 0, off = 0 ; i < args_len ; off += vob_args[i].length, i++)
		{
			memcpy(blocos[i], buf + off, vob_args[i].length);
			blocos[i][vob_args[i].length] = 0;
		}
	}
	else if (len == (linha_len - 3))	// barras
	{
	}
	else
		return("{\"erro\":98}");

	if ((i = vob_checkin(blocos)))
		return("{\"erro\":99}");

	jfields = vob_get_banco(blocos);
	if_free(buf);
	buf = json_serialize(jfields);
	json_value_free(jfields);

	return(buf);
}
// ////////////////////////////////////////////////////////////


#ifdef STANDALONE
// ////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	JSON_VALUE *jbanco;
	char *txt;
	int i;

	fprintf(stdout, "Content-Type: text/plain\n\n");

	if ((argc < args_len) && (argc != 2))
	{
		fprintf(stderr, "iFractal Desenvolvimento de Software\n");
		fprintf(stderr, "Verificacao Online de Boletos - VOB\n\n");
		fprintf(stderr, "Uso:\n\tshell$ %s", argv[0]);

		for (i = 1 ; i < args_len ; i++)
			fprintf(stderr, " %s", vob_args[i].name);

		fprintf(stderr, "\n\nExemplo:\n\tshell$ %s 00190 00009 01234 567897 01234 567012 8 62720000037165\n\n", argv[0]);
		fprintf(stdout, "{'erro':1, 'descricao':'Faltam dados.'}\n");

		return(1);
	}

	if (argc == 2)
	{
		fprintf(stdout, "%s\n", vob_decode(argv[1]));
		return(0);
	}

	if ((i = vob_checkin(argv)))
	{
		fprintf(stdout, "{'erro':%d, 'descricao':'digitacao.'}\n", i);
		return(i);
	}

	jbanco = vob_get_banco(argv);
	if (jbanco == NULL)
	{
		fprintf(stderr, "Banco nao cadastrado.\n");
		fprintf(stdout, "{'erro':99, 'descricao':'Banco nao homologado.'}\n");
		return(10);
	}

	txt = json_serialize(jbanco);
	fprintf(stdout, "%s\n", txt);
	if_free(txt);

	json_value_free(jbanco);

	return(0);
}
// ////////////////////////////////////////////////////////////
#endif


